package ru.yandex.travel.api.models.hotels;

import lombok.Getter;

public class BoundingBox {
    @Getter
    private Coordinates leftDown;
    @Getter
    private Coordinates upRight;

    private BoundingBox(Coordinates leftDown, Coordinates upRight) {
        this.leftDown = leftDown;
        this.upRight = upRight;
    }

    public static BoundingBox of(Coordinates leftDown, Coordinates upRight) {
        return new BoundingBox(leftDown, upRight);
    }

    public static BoundingBox of(String s) {
        String[] parts = s.split("~", -1);
        if (parts.length != 2) {
            throw new IllegalArgumentException("Invalid BoundingBox format " + s);
        }
        return new BoundingBox(Coordinates.of(parts[0]), Coordinates.of(parts[1]));
    }

    @Override
    public String toString() {
        return String.format("%s~%s", leftDown, upRight);
    }

    @Override
    public boolean equals(Object otherObj) {
        if (!(otherObj instanceof BoundingBox)) {
            return false;
        }
        BoundingBox other = (BoundingBox)otherObj;
        return getLeftDown().equals(other.getLeftDown()) && getUpRight().equals(other.getUpRight());
    }

    public double square() {
        return (upRight.getLat() - leftDown.getLat()) * (upRight.getLon() - leftDown.getLon());
    }

    public Coordinates center() {
        Coordinates c = new Coordinates();
        c.setLon((getLeftDown().getLon() + getUpRight().getLon()) / 2d);
        c.setLat((getLeftDown().getLat() + getUpRight().getLat()) / 2d);
        return c;
    }

    public boolean contains(BoundingBox other) {
        return (getLeftDown().getLat() <= other.getLeftDown().getLat()) &&
               (getLeftDown().getLon() <= other.getLeftDown().getLon()) &&
               (getUpRight().getLat() >= other.getUpRight().getLat()) &&
               (getUpRight().getLon() >= other.getUpRight().getLon());
    }

    public boolean contains(Coordinates point) {
        return (getLeftDown().getLat() <= point.getLat()) &&
                (getLeftDown().getLon() <= point.getLon()) &&
                (getUpRight().getLat() >= point.getLat()) &&
                (getUpRight().getLon() >= point.getLon());
    }

    public BoundingBox extendByPoint(Coordinates coordinates) {
        var newLeftDown = new Coordinates();
        newLeftDown.setLon(Math.min(getLeftDown().getLon(), coordinates.getLon()));
        newLeftDown.setLat(Math.min(getLeftDown().getLat(), coordinates.getLat()));

        var newUpRight = new Coordinates();
        newUpRight.setLon(Math.max(getUpRight().getLon(), coordinates.getLon()));
        newUpRight.setLat(Math.max(getUpRight().getLat(), coordinates.getLat()));

        return BoundingBox.of(newLeftDown, newUpRight);
    }

    public BoundingBox extendByAbsoluteValue(double absoluteDiffLon, double absoluteDiffLat) {
        var newLeftDown = new Coordinates();
        newLeftDown.setLon(getLeftDown().getLon() - absoluteDiffLon);
        newLeftDown.setLat(getLeftDown().getLat() - absoluteDiffLat);

        var newUpRight = new Coordinates();
        newUpRight.setLon(getUpRight().getLon() + absoluteDiffLon);
        newUpRight.setLat(getUpRight().getLat() + absoluteDiffLat);

        return BoundingBox.of(newLeftDown, newUpRight);
    }

    public BoundingBox extendByRelativeValue(double relativeDiff, double minAbsoluteDiff) {
        double sizeLon = Math.abs(upRight.getLon() - leftDown.getLon());
        double sizeLat = Math.abs(upRight.getLat() - leftDown.getLat());

        double deltaLon = Math.max(minAbsoluteDiff, sizeLon * relativeDiff);
        double deltaLat = Math.max(minAbsoluteDiff, sizeLat * relativeDiff);

        return extendByAbsoluteValue(deltaLon, deltaLat);
    }

    public BoundingBox extendByRelativeValue(double relativeDiff) {
        return extendByRelativeValue(relativeDiff, 0);
    }

    public static BoundingBox intersection(BoundingBox box1, BoundingBox box2) {
        Coordinates leftDown = new Coordinates();
        leftDown.setLon(Math.max(box1.getLeftDown().getLon(), box2.getLeftDown().getLon()));
        leftDown.setLat(Math.max(box1.getLeftDown().getLat(), box2.getLeftDown().getLat()));
        Coordinates upRight = new Coordinates();
        upRight.setLon(Math.min(box1.getUpRight().getLon(), box2.getUpRight().getLon()));
        upRight.setLat(Math.min(box1.getUpRight().getLat(), box2.getUpRight().getLat()));
        if (leftDown.getLon() < upRight.getLon() && leftDown.getLat() < upRight.getLat()) {
            return BoundingBox.of(leftDown, upRight);
        } else {
            return BoundingBox.of(leftDown, leftDown);//empty intersection
        }
    }

    public static double intersectionOverUnion(BoundingBox box1, BoundingBox box2) {
        BoundingBox boxIntersection = BoundingBox.intersection(box1, box2);
        double intersection = boxIntersection.square();
        double union = box1.square() + box2.square() - intersection;
        if (Math.abs(union) < 1e-20) {
            return 0d;
        }
        return intersection / union;
    }
}
