package ru.yandex.travel.api.models.hotels;

import java.util.Locale;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

@Getter
@Setter
@NoArgsConstructor
@AllArgsConstructor
public class Coordinates {
    private static double EPS = 1e-7;
    private double lon; // X
    private double lat; // Y

    public static Coordinates of(String s) {
        String[] parts = s.split(",", -1);
        if (parts.length != 2) {
            throw new IllegalArgumentException("Invalid Coordinates format " + s);
        }
        Coordinates coordinates = new Coordinates();
        coordinates.setLon(parseCoordinate(parts[0], -180, 180));
        coordinates.setLat(parseCoordinate(parts[1], -90, 90));
        return coordinates;
    }

    public static Coordinates of(double lat, double lon) {
        Coordinates coordinates = new Coordinates();
        coordinates.setLon(validateCoordinate(lon, -180, 180));
        coordinates.setLat(validateCoordinate(lat, -90, 90));
        return coordinates;
    }

    private static double parseCoordinate(String strVal, double min, double max) {
        double doubleVal = Double.parseDouble(strVal);
        return validateCoordinate(doubleVal, min, max);
    }

    private static double validateCoordinate(double val, double min, double max) {
        if (!Double.isFinite(val)) {
            throw new IllegalArgumentException("Invalid coordinate " + val);
        }
        if (val < min || val > max) {
            throw new IllegalArgumentException("Coordinate out of bound: " + val);
        }
        return val;
    }

    @Override
    @SuppressWarnings("MethodDoesntCallSuperMethod")
    public Coordinates clone() {
        Coordinates coordinates = new Coordinates();
        coordinates.setLon(getLon());
        coordinates.setLat(getLat());
        return coordinates;
    }

    @Override
    public String toString() {
        return String.format(Locale.US, "%f,%f", lon, lat);
    }

    @Override
    public boolean equals(Object otherObj) {
        if (!(otherObj instanceof Coordinates)) {
            return false;
        }
        Coordinates other = (Coordinates)otherObj;
        return Math.abs(getLon() - other.getLon()) < EPS && Math.abs(getLat() - other.getLat()) < EPS;
    }

    @Override
    public int hashCode() {
        throw new UnsupportedOperationException("No compatible with equals hashCode implementation is available");
    }

    public static double distance(Coordinates c1, Coordinates c2) {
        return Math.sqrt(Math.pow(c1.getLon() - c2.getLon(), 2) + Math.pow(c1.getLat() - c2.getLat(), 2));
    }
}
