package ru.yandex.travel.api.models.hotels;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import lombok.Data;

import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.offercache.api.TReadResp;

@Data
public class OfferCacheMetadata {
    private boolean isFake;
    private boolean useProdOfferCache;
    private Price.Currency currency;
    private Map<String, String> pansionTypes;
    private Map<String, OperatorInfo> operatorById;
    private Map<Integer, PartnerInfo> partnerById;
    private OfferSearchParams offerSearchParams;
    private OfferSearchProgress progress;

    private OfferCacheMetadata() {
    }

    public static OfferCacheMetadata extractFromProto(TReadResp readResp, boolean useProdOfferCache) {
        OfferCacheMetadata ocMeta = new OfferCacheMetadata();
        ocMeta.setFake(false);
        ocMeta.setUseProdOfferCache(useProdOfferCache);
        // Currency
        switch (readResp.getCurrency()) {
            case RUB:
                ocMeta.setCurrency(Price.Currency.RUB);
                break;
            case USD:
                ocMeta.setCurrency(Price.Currency.USD);
                break;
            default:
                throw new RuntimeException("Got unknown currency from OC: " + readResp.getCurrency());
        }
        // pansionTypes
        Map<String, String> ptMap = new HashMap<>();
        for (Map.Entry<String, TReadResp.TPansionInfo> p : readResp.getPansionsMap().entrySet()) {
            ptMap.put(p.getKey(), p.getValue().getName());
        }
        ocMeta.setPansionTypes(ptMap);

        // operatorById
        Map<String, OperatorInfo> opById = new HashMap<>();
        for (Map.Entry<Integer, TReadResp.TOperatorInfo> entry : readResp.getOperatorsMap().entrySet()) {
            OperatorInfo operatorInfo = new OperatorInfo();
            operatorInfo.setId(String.valueOf(entry.getKey()));
            operatorInfo.setName(entry.getValue().getName());
            operatorInfo.setGreenUrl(entry.getValue().getGreenUrl());
            operatorInfo.setIconUrl(entry.getValue().getFaviconUrl());
            operatorInfo.setBookOnYandex(entry.getValue().getIsBookOnYandex());
            opById.put(operatorInfo.getId(), operatorInfo);
        }
        ocMeta.setOperatorById(opById);

        // partnerById
        Map<Integer, PartnerInfo> partnerById = new HashMap<>();
        for (Map.Entry<Integer, TReadResp.TPartnerInfo> entry : readResp.getPartnersMap().entrySet()) {
            PartnerInfo partnerInfo = new PartnerInfo();
            partnerInfo.setId(entry.getKey());
            partnerInfo.setBookOnYandex(entry.getValue().getIsBookOnYandex());
            partnerById.put(partnerInfo.getId(), partnerInfo);
        }
        ocMeta.setPartnerById(partnerById);

        // offerSearchParams
        OfferSearchParams osp = new OfferSearchParams();
        osp.setCheckinDate(LocalDate.parse(readResp.getDate()));
        osp.setCheckoutDate(LocalDate.parse(readResp.getCheckOutDate()));
        Ages ages = Ages.fromString(readResp.getAges());
        osp.setAdults(ages.getAdults());
        osp.setChildrenAges(ages.getChildrenAges());
        ocMeta.setOfferSearchParams(osp);

        // progress
        OfferSearchProgress progress = new OfferSearchProgress();
        progress.setFinished(readResp.getIsFinished());
        progress.setPartnersTotal(readResp.getProgress().getOperatorsTotal());
        progress.setPartnersComplete(readResp.getProgress().getOperatorsComplete());
        progress.setFinishedPartners(readResp.getProgress().getFinishedPartnersList());
        progress.setPendingPartners(readResp.getProgress().getPendingPartnersList());
        ocMeta.setProgress(progress);

        return ocMeta;
    }

    public static OfferCacheMetadata createFake(OfferSearchParamsProvider paramsProvider) {
        OfferCacheMetadata ocMeta = new OfferCacheMetadata();
        ocMeta.setFake(true);
        ocMeta.setUseProdOfferCache(false);
        // Currency
            ocMeta.setCurrency(Price.Currency.RUB);
        // pansionTypes
        ocMeta.setPansionTypes(new HashMap<>());

        // operatorById
        ocMeta.setOperatorById(new HashMap<>());

        // partnerById
        ocMeta.setPartnerById(new HashMap<>());

        // offerSearchParams
        OfferSearchParams osp = new OfferSearchParams();
        osp.setCheckinDate(paramsProvider.getCheckinDate());
        osp.setCheckoutDate(paramsProvider.getCheckoutDate());
        if (osp.getCheckinDate() == null) {
            if (osp.getCheckoutDate() == null) {
                osp.setCheckinDate(LocalDate.now());
                osp.setCheckoutDate(osp.getCheckinDate().plusDays(1));
            } else {
                osp.setCheckinDate(osp.getCheckoutDate().minusDays(1));
            }
        } else {
            if (osp.getCheckoutDate() == null) {
                osp.setCheckoutDate(osp.getCheckinDate().plusDays(1));
            }
        }
        osp.setAdults(Objects.requireNonNullElse(paramsProvider.getAdults(), 2));
        if (paramsProvider.getChildrenAges() == null) {
            osp.setChildrenAges(new ArrayList<>());
        } else {
            osp.setChildrenAges(paramsProvider.getChildrenAges());
        }
        ocMeta.setOfferSearchParams(osp);

        // progress
        OfferSearchProgress progress = new OfferSearchProgress();
        progress.setFinished(true);
        progress.setPartnersTotal(0);
        progress.setPartnersComplete(0);
        progress.setFinishedPartners(Collections.emptyList());
        progress.setPendingPartners(Collections.emptyList());
        ocMeta.setProgress(progress);

        return ocMeta;
    }

    public static OfferCacheMetadata extract(TReadResp readResp, OfferSearchParamsProvider paramsProvider, boolean useProdOfferCache) {
        if (readResp == null) {
            return createFake(paramsProvider);
        } else {
            return extractFromProto(readResp, useProdOfferCache);
        }
    }

}
