package ru.yandex.travel.api.models.hotels;

import java.util.Base64;
import java.util.UUID;

import javax.annotation.Nullable;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils;
import ru.yandex.travel.api.proto.hotels_portal.TSearchContext;

@Slf4j
public class SearchContext {
    private String pollingId;
    private TSearchContext.Builder proto;

    public static SearchContext parse(@Nullable String txt) throws InvalidProtocolBufferException {
        SearchContext res = new SearchContext();
        if (txt == null) {
            res.proto = TSearchContext.newBuilder();
        } else {
            int delimeterPos = txt.indexOf('~');
            if (delimeterPos < 0) {
                throw new RuntimeException("Invalid context");
            }
            res.pollingId = txt.substring(0, delimeterPos);
            res.proto = TSearchContext.parseFrom(HotelsPortalUtils.decodeBytesFromString(txt.substring(delimeterPos + 1))).toBuilder();
            if (res.proto.hasGeoSearchCtxBytes()) {
                res.proto.setGeoSearchCtxStr(Base64.getEncoder().encodeToString(res.proto.getGeoSearchCtxBytes().toByteArray()));
            }
        }
        return res;
    }

    public String getPollingId() {
        return pollingId;
    }

    public TSearchContext.Builder protoBuilder() {
        return proto;
    }

    public void generateNewPollingId() {
        // RANDOMRANDOM-xx-yy
        // xx - move bbox
        // yy - navigation
        pollingId = UUID.randomUUID().toString().replace("-", "") + "-0-0";
    }

    public void incrementPollingIdOnNavigation() {
        String[] parts = pollingId.split("-");
        if (parts.length >= 3) {
            parts[2] = Integer.toString(Integer.valueOf(parts[2]) + 1);
        }
        pollingId = String.join("-", parts);
    }

    public void incrementPollingIdOnBboxChange() {
        String[] parts = pollingId.split("-");
        if (parts.length >= 2) {
            parts[1] = Integer.toString(Integer.valueOf(parts[1]) + 1);
        }
        if (parts.length >= 3) {
            parts[2] = "0";
        }
        pollingId = String.join("-", parts);
    }

    public void setOriginalBbox(BoundingBox bbox) {
        proto.clearOriginalBbox();
        proto.addOriginalBbox(bbox.getLeftDown().getLon());
        proto.addOriginalBbox(bbox.getLeftDown().getLat());
        proto.addOriginalBbox(bbox.getUpRight().getLon());
        proto.addOriginalBbox(bbox.getUpRight().getLat());
    }

    public void clearOriginalBbox() {
        proto.clearOriginalBbox();
    }

    public boolean hasOriginalBbox() {
        return proto.getOriginalBboxCount() == 4;
    }

    public BoundingBox getOriginalBbox() {
        if (proto.getOriginalBboxCount() != 4) {
            return null;
        }
        Coordinates leftDown = new Coordinates();
        leftDown.setLon(proto.getOriginalBbox(0));
        leftDown.setLat(proto.getOriginalBbox(1));
        Coordinates upRight = new Coordinates();
        upRight.setLon(proto.getOriginalBbox(2));
        upRight.setLat(proto.getOriginalBbox(3));
        return BoundingBox.of(leftDown, upRight);
    }

    public String serialize() {
        if (proto.hasGeoSearchCtxStr()) {
            try {
                proto.setGeoSearchCtxBytes(ByteString.copyFrom(Base64.getDecoder().decode(proto.getGeoSearchCtxStr())));
            } catch (Exception exc) {
                log.info("Failed to parse geosearch context as b64: {}", proto.getGeoSearchCtxStr(), exc);
            }
        }
        return pollingId + "~" + HotelsPortalUtils.encodeProtoToString(proto.build());
    }

    @Override
    public String toString() {
        return pollingId + " " + proto.toString().replace("\n", " ");
    }
}
