package ru.yandex.travel.api.services.avia.country_restrictions;

import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.TestOnly;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.models.avia.country_restrictions.v1.ExtendedMetricsAnswer;
import ru.yandex.travel.api.services.geo.model.PointInfo;
import ru.yandex.travel.proto.avia.TCountryRestrictionsV1;
import ru.yandex.travel.yt_lucene_index.MapPersistentConfig;

@Component
@EnableConfigurationProperties(CountryRestrictionsExtendedServiceV1Properties.class)
@Slf4j
public class CountryRestrictionsExtendedServiceV1 {
    private final CountryRestrictionsMeters meters;
    private final MapPersistentConfig<Long, ExtendedMetricsAnswer, TCountryRestrictionsV1> persistentConfig;
    private final ObjectMapper mapper = new ObjectMapper().setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);

    public CountryRestrictionsExtendedServiceV1(CountryRestrictionsExtendedServiceV1Properties params) {
        meters = new CountryRestrictionsMeters();
        if (params.isEnabled()) {
            persistentConfig = new MapPersistentConfig<>(params, "CountryRestrictionsExtendedV1",
                    TCountryRestrictionsV1::newBuilder, TCountryRestrictionsV1::getKey, this::valueGetter);
        } else {
            persistentConfig = null;
        }
    }

    @TestOnly
    protected ExtendedMetricsAnswer valueGetter(TCountryRestrictionsV1 v) {
        try {
            return mapper.readValue(v.getValue(), ExtendedMetricsAnswer.class);
        } catch (JsonProcessingException e) {
            meters.getV1JsonParseErrorsOther().increment();
            log.error("CountryRestrictionsExtendedService error: ", e);
            return null;
        }
    }

    private void ensureEnabled() {
        if (persistentConfig == null) {
            throw new IllegalStateException("CountryRestrictionsV1 service is disabled");
        }
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }

    private ExtendedMetricsAnswer getByGeoId(Long geoId) {
        if (persistentConfig == null) {
            return null;
        }
        return persistentConfig.getByKey(geoId);
    }

    private ExtendedMetricsAnswer getByPointInfo(PointInfo pointInfo) {
        return getByGeoId(Long.valueOf(pointInfo.getGeoId()));
    }

    public ExtendedMetricsAnswer getByPointInfoHierarchy(List<PointInfo> hierarchy) {
        ExtendedMetricsAnswer restrictions = null;

        for (PointInfo pointInfo : hierarchy) {
            restrictions = getByPointInfo(pointInfo);
            if (restrictions != null) {
                break;
            }
        }

        return restrictions;
    }
}
