package ru.yandex.travel.api.services.avia.country_restrictions;

import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.TestOnly;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.models.avia.country_restrictions.v1.Metric;
import ru.yandex.travel.api.models.avia.country_restrictions.v1.ReturnPointInfo;
import ru.yandex.travel.api.services.geo.model.PointInfo;
import ru.yandex.travel.proto.avia.TCountryRestrictionsV1;
import ru.yandex.travel.yt_lucene_index.MapPersistentConfig;


@Component
@EnableConfigurationProperties(CountryRestrictionsServiceV1Properties.class)
@Slf4j
public class CountryRestrictionsServiceV1 {
    private final CountryRestrictionsMeters meters;
    private MapPersistentConfig<Long, Map<String, Metric>, TCountryRestrictionsV1> persistentConfig = null;
    private final ObjectMapper mapper = new ObjectMapper().setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);

    public CountryRestrictionsServiceV1(CountryRestrictionsServiceV1Properties params) {
        meters = new CountryRestrictionsMeters();
        if (params.isEnabled()) {
            persistentConfig = new MapPersistentConfig<>(params, "CountryRestrictionsV1",
                    TCountryRestrictionsV1::newBuilder, TCountryRestrictionsV1::getKey, this::valueGetter);
        }
    }

    @TestOnly
    protected Map<String, Metric> valueGetter(TCountryRestrictionsV1 v) {
        try {
            Map<String, Metric> metrics = mapper.readValue(v.getValue(), new TypeReference<>() {});
            for (Map.Entry<String, Metric> entry : metrics.entrySet()) {
                Metric m = entry.getValue();
                if (m.getValue() == null || m.getText() == null || m.getAdditions() == null || m.getExclusions() == null) {
                    meters.getV1JsonParseErrorsNoRequiredFields().increment();
                    log.error("CountryRestrictionsService error: no required field in parsed Metric");
                    return null;
                }
            }

            return metrics;
        } catch (JsonProcessingException e) {
            meters.getV1JsonParseErrorsOther().increment();
            log.error("CountryRestrictionsService error: ", e);
            return null;
        }
    }

    private void ensureEnabled() {
        if (persistentConfig == null) {
            throw new IllegalStateException("CountryRestrictionsV1 service is disabled");
        }
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }

    private Map<String, Metric> getByGeoId(Long geoId) {
        if (persistentConfig == null) {
            return null;
        }
        return persistentConfig.getByKey(geoId);
    }

    private Map<String, Metric> getByPointInfo(PointInfo pointInfo) {
        var restrictions = getByGeoId(Long.valueOf(pointInfo.getGeoId()));
        if (restrictions == null) {
            return null;
        }

        ReturnPointInfo returnPointInfo = new ReturnPointInfo(
                pointInfo.getKey(), Long.valueOf(pointInfo.getGeoId()),
                pointInfo.getType().ordinal(), pointInfo.getLinguistics()
        );

        for (String key : restrictions.keySet()) {
            restrictions.get(key).setPointInfo(returnPointInfo);
        }

        return restrictions;
    }

    public Map<String, Metric> getByPointInfoHierarchy(List<PointInfo> hierarchy) {
        Map<String, Metric> restrictions = null;

        for (PointInfo pointInfo : hierarchy) {
            restrictions = getByPointInfo(pointInfo);
            if (restrictions != null) {
                break;
            }
        }

        return restrictions;
    }
}
