package ru.yandex.travel.api.services.avia.fares;

import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.model.search.Flight;
import ru.yandex.avia.booking.partners.gateways.model.search.Segment;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.travel.api.services.avia.references.AviaFlightGeoData;
import ru.yandex.travel.api.services.avia.references.AviaGeoDataService;

@Service
@RequiredArgsConstructor
public class AviaFareFamilyInputFactory {
    private final AviaGeoDataService geoDataService;

    public Document convertVariant(Variant variant, Map<String, String> fareCodes) {
        List<Flight> allFlights = new ArrayList<>();
        for (Segment segment : variant.getSegments()) {
            allFlights.addAll(segment.getFlights());
        }
        // Aeroflot-specific logic, should be generalized later
        if (allFlights.size() > 4) {
            throw new AviaFareRulesException("Multiple tickets aren't supported; offerId=" + variant.getPriceInfo().getId());
        }
        return convertSegment(variant.getCountryOfSale(), allFlights, fareCodes);
    }

    private Document convertSegment(String countryOfSale, List<Flight> flights, Map<String, String> fareCodes) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(countryOfSale), "countryOfSale can not be null or empty");
        Document doc = DocumentHelper.createDocument();
        Element root = doc.addElement("Leg");
        root.addElement("CountryOfSale").setText(countryOfSale);

        for (Flight flight : flights) {
            Element seg = root.addElement("Seg");
            seg.addElement("Id").setText(flight.getId());
            String fareCode = fareCodes.get(flight.getId());
            Preconditions.checkArgument(!Strings.isNullOrEmpty(fareCode), "No fare code fot the %s segment", flight.getId());
            // aeroflot-specific part, we need to generalize it somehow when there are more partners
            String fareCodePrefix = fareCode.length() > 3 ? fareCode.substring(0, 3) : fareCode;
            seg.addElement("FareCodePrefix").setText(fareCodePrefix);

            AviaFlightGeoData geoData = geoDataService.getFlightGeoData(flight);
            seg.addElement("FromAirport").setText(geoData.getFromAirportCode());
            seg.addElement("FromCountry").setText(geoData.getFromCountryCode());
            seg.addElement("ToAirport").setText(geoData.getToAirportCode());
            seg.addElement("ToCountry").setText(geoData.getToCountryCode());

            // xpath 1.0 doesn't support strings comparison, we have to compare raw time seconds
            seg.addElement("DepartureTs").setText(String.valueOf(flight.getDepartureDateTime().toEpochSecond(ZoneOffset.UTC)));
            seg.addElement("ArrivalTs").setText(String.valueOf(flight.getArrivalDateTime().toEpochSecond(ZoneOffset.UTC)));
        }

        return doc;
    }
}
