package ru.yandex.travel.api.services.avia.fares;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import org.dom4j.Document;
import org.dom4j.Node;

import ru.yandex.avia.booking.ff.model.FareFamily;
import ru.yandex.avia.booking.partners.gateways.aeroflot.parsing.XmlUtils;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

@Slf4j
public class AviaFareFamilyReferenceParser {
    public static List<FareFamily> loadFareFamilies(String fareFamiliesFile) {
        ObjectMapper jsonMapper = new ObjectMapper()
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        TypeReference<List<FareFamily>> familiesType = new TypeReference<>() {
        };
        List<FareFamily> fareFamilies = loadResource(fareFamiliesFile,
                is -> jsonMapper.readValue(is, familiesType));
        log.info("Loaded tariffs: src={}, values={}", fareFamiliesFile, fareFamilies.stream()
                .map(ff -> ff.getTariffGroupName().getForLang("ru")).collect(toList()));
        return ImmutableList.copyOf(fareFamilies);
    }

    public static Map<String, String> loadExternalExpressions(String externalExpressionsFile) {
        Document expressions = loadResource(externalExpressionsFile, XmlUtils::parseXml);
        Map<String, String> externalExpressions = ImmutableMap.copyOf(
                expressions.selectNodes("Expressions/Expression").stream()
                        .collect(toMap(e -> e.valueOf("@id"), Node::getText)));
        log.info("Loaded external xpath expressions: src={}, values={}",
                externalExpressionsFile, externalExpressions.keySet());
        return externalExpressions;
    }

    private static <T> T loadResource(String resource, ResourceConverter<T> converter) {
        try (InputStream is = AviaFareFamilyReferenceParser.class.getClassLoader().getResourceAsStream(resource)) {
            return converter.convert(is);
        } catch (IOException e) {
            throw new RuntimeException("Failed to convert the resource: " + resource, e);
        }
    }

    private interface ResourceConverter<T> {
        T convert(InputStream is) throws IOException;
    }
}
