package ru.yandex.travel.api.services.avia.orders;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.grpc.Deadline;
import io.opentracing.Span;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.BookingGateway;
import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotGateway;
import ru.yandex.avia.booking.partners.gateways.model.PayloadMapper;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayload;
import ru.yandex.avia.booking.service.dto.AeroflotStateDTO;
import ru.yandex.avia.booking.service.dto.CompositeOrderStateDTO;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.avia.booking.service.dto.OrderListDTO;
import ru.yandex.avia.booking.service.dto.form.CreateOrderForm;
import ru.yandex.avia.booking.service.dto.form.InitOrderPaymentForm;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.services.avia.AviaBookingMeters;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.AppCallIdGenerator;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.proto.EDeviceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetAeroflotStateReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TListOrdersReq;
import ru.yandex.travel.orders.proto.TListOrdersRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderInvoiceInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.tracing.TracerHelpers;
import ru.yandex.travel.workflow.EWorkflowState;

import static java.util.stream.Collectors.toList;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
@Slf4j
public class AviaOrchestratorClientAdapter {
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final AeroflotGateway aeroflotGateway;
    private final AviaOrchestratorModelConverter modelConverter;
    private final ApiTokenEncrypter tokenEncrypter;
    private final AviaBookingMeters meters;
    private final Tracer tracer;

    private Map<EServiceType, BookingGateway> providerTypeToGateway;

    @PostConstruct
    public void init() {
        providerTypeToGateway = ImmutableMap.of(
                EServiceType.PT_FLIGHT, aeroflotGateway
        );
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub orchestratorChannel() {
        return orchestratorClientFactory.createFutureStubForAvia();
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub slavePreferredOrchestratorChannel() {
        return orchestratorClientFactory.createFutureStubForAviaReadOnly();
    }

    public CompletableFuture<OrderDTO> createOrder(CreateOrderForm form, ServicePayload payload,
                                                   UserCredentials userCredentials, TAviaTestContext testContext) {
        Span span = tracer.activeSpan();
        log.info("sending createOrder request to the orchestrator: variant_token={}", form.getVariantToken());
        TCreateOrderReq.Builder createReq = TCreateOrderReq.newBuilder()
                .setDeduplicationKey(AppCallIdGenerator.KEY.get().generateUUID("createOrder"))
                .setOrderType(AviaOrchestratorEnumsMapper.lookup(
                        AviaOrchestratorEnumsMapper.PARTNER_ID_TO_ORDER_TYPE, payload.getPartnerId()))
                .addCreateServices(createServiceReq(form, payload, testContext))
                .setCurrency(AviaOrchestratorEnumsMapper.lookup(
                        AviaOrchestratorEnumsMapper.CURRENCY_JDK_TO_PROTO, payload.getPreliminaryCost().getCurrency()))
                .setOwner(createUserInfo(form, userCredentials));
        createReq.setLabel(Strings.nullToEmpty(form.getMarker()));
        if (!Strings.isNullOrEmpty(form.getPaymentTestContextToken())) {
            createReq.setPaymentTestContext(tokenEncrypter.fromPaymentTestContextToken(form.getPaymentTestContextToken()));
            createReq.setMockPayment(true);
        }
        OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub ordersClient = orchestratorChannel();
        return FutureUtils.buildCompletableFuture(ordersClient.createOrder(createReq.build()))
                .thenCompose(TracerHelpers.applyWithActiveSpanScope(tracer, span, createRsp -> {
                    meters.getOrdersCreated().increment();

                    TOrderInfo order = createRsp.getNewOrder();
                    ensureExpectedOrderStructure(order);
                    String orderId = order.getOrderId();
                    String serviceId = order.getService(0).getServiceId();
                    log.info("the order has been created: order_id={} service_id={}", orderId, serviceId);
                    log.trace("TCreateOrderRsp: {}", createRsp);

                    log.info("reserving the order: order_id={}", orderId);
                    TReserveReq reserveReq = TReserveReq.newBuilder()
                            .setCallId(callIdForMethod("reserveOrder"))
                            .setOrderId(orderId).build();
                    return FutureUtils.buildCompletableFuture(ordersClient.reserve(reserveReq))
                            .thenApply(reserveRsp -> modelConverter.fromProto(order, payload.getClass()));
                }));
    }

    public CompletableFuture<ArrayList<AeroflotStateDTO>> getAeroflotState(List<UUID> orderIds) {
        //запрос до аэрофлота может выполняться долго (50 секунд), у grpc разрыв после 60 секунд - установил 2 минуты
        OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub ordersClient =
                orchestratorChannel().withDeadline(Deadline.after(2, TimeUnit.MINUTES));
        var completableFutures = orderIds.stream().map(
                orderId -> FutureUtils.buildCompletableFuture(ordersClient.getOrderAeroflotState(TGetAeroflotStateReq.newBuilder()
                        .setOrderId(orderId.toString()).build()))
        ).collect(toList());

        var res = new ArrayList<AeroflotStateDTO>();
        try {
            var allFutures =
                    CompletableFuture.allOf(completableFutures.toArray(new CompletableFuture[completableFutures.size()]));
            var allCompletableFutures = allFutures.thenApply(future -> completableFutures.stream()
                    .map(CompletableFuture::join)
                    .collect(Collectors.toList())).get();

            for (var rsp : allCompletableFutures) {
                var o = new AeroflotStateDTO();
                o.setOrderId(rsp.getOrderState().getOrderId());
                o.setState(rsp.getOrderState().getState());
                res.add(o);
            }
        } catch (Exception e) {
            log.error("", e);
        }
        return CompletableFuture.completedFuture(res);
    }

    private TCreateServiceReq createServiceReq(CreateOrderForm form, ServicePayload payload,
                                               TAviaTestContext testContext) {
        TCreateServiceReq.Builder createServiceReqBuilder = TCreateServiceReq.newBuilder()
                .setServiceType(AviaOrchestratorEnumsMapper.lookup(
                        AviaOrchestratorEnumsMapper.PARTNER_ID_TO_PROVIDER_TYPE, payload.getPartnerId()))
                .setSourcePayload(TJson.newBuilder()
                        .setValue(PayloadMapper.toJson(payload))
                        .build());
        if (testContext != null) {
            createServiceReqBuilder.setAviaTestContext(testContext);
        } else {
            if (!Strings.isNullOrEmpty(form.getTestContextToken())) {
                createServiceReqBuilder.setAviaTestContext(tokenEncrypter.fromAviaTestContextToken(form.getTestContextToken()));
            }
        }
        return createServiceReqBuilder.build();
    }

    private TUserInfo createUserInfo(CreateOrderForm form, UserCredentials userCredentials) {
        TUserInfo.Builder userInfo = TUserInfo.newBuilder();
        userInfo.setYandexUid(userCredentials.getYandexUid());
        userInfo.setEmail(form.getEmail());
        userInfo.setPhone(form.getPhone());
        if (userCredentials.getPassportId() != null) {
            userInfo.setPassportId(userCredentials.getPassportId());
        }
        if (userCredentials.getLogin() != null) {
            userInfo.setLogin(userCredentials.getLogin());
        }
        userInfo.setIp(form.getUserIp());
        return userInfo.build();
    }

    private String deviceTypeToSource(EDeviceType deviceType) {
        if (deviceType == null || deviceType == EDeviceType.DT_DESKTOP) {
            return "desktop";
        }
        if (deviceType == EDeviceType.DT_MOBILE) {
            return "mobile";
        }
        throw new IllegalArgumentException("Unsupported device type: " + deviceType);
    }

    public CompletableFuture<CompositeOrderStateDTO> getState(UUID orderId, boolean preferSlave) {
        CompletableFuture<TOrderInfo> orderInfoFuture;
        if (preferSlave) {
            orderInfoFuture = getOrchestratorOrderRO(orderId);
        } else {
            orderInfoFuture = getOrchestratorOrder(orderId);
        }
        return orderInfoFuture.thenApply(order -> {
            ensureExpectedOrderStructure(order);
            TServiceInfo service = order.getService(0).getServiceInfo();
            TOrderInvoiceInfo invoice = order.getInvoiceCount() == 1 ? order.getInvoice(0) : null;
            return CompositeOrderStateDTO.builder()
                    .orderState(order.getWorkflowState() != EWorkflowState.WS_CRASHED ?
                            order.getAeroflotOrderState() : EAeroflotOrderState.OS_CANCELLED)
                    .serviceState(service.getAeroflotItemState())
                    .invoiceState(invoice != null ? invoice.getAeroflotInvoiceState() : null)
                    .paymentUrl(invoice != null ? invoice.getPaymentUrl() : null)
                    .confirmationUrl(invoice != null ? invoice.getConfirmationUrl() : null)
                    .displayOrderState(order.getDisplayOrderState())
                    .build();
        });
    }

    public CompletableFuture<TOrderInfo> getOrchestratorOrder(UUID orderId) {
        TGetOrderInfoReq orderInfoReq = TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId.toString())
                .build();
        return FutureUtils.buildCompletableFuture(orchestratorChannel().getOrderInfo(orderInfoReq))
                .thenApply(TGetOrderInfoRsp::getResult);
    }

    private CompletableFuture<TOrderInfo> getOrchestratorOrderRO(UUID orderId) {
        TGetOrderInfoReq orderInfoReq = TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId.toString())
                .build();
        return FutureUtils.buildCompletableFuture(
                        slavePreferredOrchestratorChannel()
                                .getOrderInfo(orderInfoReq))
                .thenApply(TGetOrderInfoRsp::getResult);
    }

    public CompletableFuture<OrderDTO> getOrderInfo(UUID orderId) {
        return getOrchestratorOrder(orderId)
                .thenApply(this::convertOrderInfo);
    }

    private OrderDTO convertOrderInfo(TOrderInfo order) {
        ensureExpectedOrderStructure(order);

        EServiceType providerType = order.getService(0).getServiceType();
        BookingGateway gateway = AviaOrchestratorEnumsMapper.lookup(providerTypeToGateway, providerType);
        return modelConverter.fromProto(order, gateway.getPayloadType());
    }

    public CompletableFuture<Void> startPayment(UUID orderId, InitOrderPaymentForm params) {
        log.info("Starting payment from order {}", orderId);
        return getOrchestratorOrder(orderId).thenCompose(order -> {
            ensureExpectedOrderStructure(order);
            TOrderServiceInfo service = order.getService(0);
            EServiceType providerType = service.getServiceType();
            BookingGateway gateway = AviaOrchestratorEnumsMapper.lookup(providerTypeToGateway, providerType);
            Class<? extends ServicePayload> payloadType = gateway.getPayloadType();
            ServicePayload payload = PayloadMapper.fromJson(service.getServiceInfo().getPayload().getValue(),
                    payloadType);

            return FutureUtils.buildCompletableFuture(orchestratorChannel().startPayment(TStartPaymentReq.newBuilder()
                            .setCallId(callIdForMethod("startPayment"))
                            .setOrderId(orderId.toString())
                            .setSource(deviceTypeToSource(params.getDeviceType()))
                            .setReturnUrl(params.getPaymentRedirectUrl())
                            .setConfirmationReturnUrl(params.getConfirmationRedirectUrl())
                            .setInvoiceType(AviaOrchestratorEnumsMapper.lookup(
                                    AviaOrchestratorEnumsMapper.PARTNER_ID_TO_INVOICE_TYPE, payload.getPartnerId()
                            ))
                            .setNewCommonPaymentWebForm(params.isEnableNewCommonPaymentForm())
                            .build()))
                    .thenApply(paymentRsp -> null);
        });
    }

    private void ensureExpectedOrderStructure(TOrderInfo order) {
        Preconditions.checkArgument(order.getServiceCount() == 1,
                "exactly 1 service is expected instead of %s", order.getServiceCount());
        Preconditions.checkArgument(order.getInvoiceCount() <= 1,
                "no more than 1 invoice is expected instead of %s", order.getInvoiceCount());
    }

    public CompletableFuture<OrderListDTO> listOrders(int offset, int limit, List<EDisplayOrderState> states) {
        TListOrdersReq.Builder rqBuilder = TListOrdersReq.newBuilder()
                .addTypes(EDisplayOrderType.DT_AVIA)
                .setPage(TOffsetPage.newBuilder()
                        .setOffset(offset)
                        .setLimit(limit)
                        .build());
        if (states != null) {
            if (states.size() >= 1) {
                rqBuilder.setDisplayOrderState(states.get(0));
            }
            rqBuilder.addAllDisplayOrderStates(states);
        }
        CompletableFuture<TListOrdersRsp> forders =
                FutureUtils.buildCompletableFuture(
                        orchestratorChannel().listOrders(rqBuilder.build()));
        return forders.thenApply(orders -> {
                    OrderListDTO res = new OrderListDTO();
                    res.setOffset(orders.getPage().getOffset());
                    res.setLimit(orders.getPage().getLimit());
                    res.setHasMoreOrders(orders.getHasMoreOrders());
                    res.setOrders(orders.getOrderInfoList().stream()
                            .map(this::convertOrderInfo)
                            .collect(toList()));
                    return res;
                }
        );
    }

    private String callIdForMethod(String methodName) {
        return AppCallIdGenerator.KEY.get().generate(methodName);
    }
}
