package ru.yandex.travel.api.services.avia.orders;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.ff.FareTermsHelper;
import ru.yandex.avia.booking.ff.model.SegmentFare;
import ru.yandex.avia.booking.partners.gateways.aeroflot.converter.AeroflotVariantConverter;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.model.PayloadMapper;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingPriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.PassengerPricePart;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.avia.booking.service.commons.OrderState;
import ru.yandex.avia.booking.service.dto.AirReservationDTO;
import ru.yandex.avia.booking.service.dto.FlightDTO;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.avia.booking.service.dto.SegmentDTO;
import ru.yandex.avia.booking.service.dto.TravellerInfoDTO;
import ru.yandex.avia.booking.service.dto.TravellerPriceInfoDTO;
import ru.yandex.travel.api.services.avia.fares.AviaFareFamilyService;
import ru.yandex.travel.api.services.avia.references.AirlineLoyaltyProgramCodeMapper;
import ru.yandex.travel.api.services.avia.references.AviaReferenceJsonFactory;
import ru.yandex.travel.api.services.avia.variants.AviaVariantDTOFactory;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirlineDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TUserInfo;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.api.services.avia.orders.AviaOrchestratorEnumsMapper.SERVICE_ITEM_STATE_TO_ERROR_CODE;
import static ru.yandex.travel.api.services.avia.orders.AviaOrderService.NONAME_MIDDLENAME;

@Service
@AllArgsConstructor
public class AviaOrchestratorModelConverter {
    private final AviaAirportDictionary airportCacheService;
    private final AviaAirlineDictionary airlineCacheService;
    private final AviaFareFamilyService fareFamilyService;
    private final AirlineLoyaltyProgramCodeMapper airlineLoyaltyProgramCodeMapper;
    private final AviaReferenceJsonFactory referenceJsonFactory;
    private final AviaVariantDTOFactory aviaVariantDTOFactory;

    private static TravellerPriceInfoDTO convertPricePart(PassengerPricePart ppp) {
        return TravellerPriceInfoDTO.builder()
                .fare(ppp.getFare())
                .tax(ppp.getTax())
                .total(ppp.getTotal())
                .build();
    }

    public OrderDTO fromProto(TOrderInfo info, Class<? extends ServicePayload> payloadType) {
        OrderDTO order = new OrderDTO();

        order.setId(info.getOrderId());
        order.setPrettyId(info.getPrettyId());
        order.setState(AviaOrchestratorEnumsMapper.lookup(
                AviaOrchestratorEnumsMapper.ORDER_STATE_PROTO_TO_API, info.getAeroflotOrderState(),
                OrderState.UNRECOGNIZED));

        if (info.hasExpiresAt()) {
            order.setTimeLimitAt(ProtoUtils.toLocalDateTime(info.getExpiresAt()));
        }

        TUserInfo userInfo = info.getOwner();
        order.setEmail(userInfo.getEmail());
        order.setPhone(userInfo.getPhone());

        Preconditions.checkArgument(info.getServiceCount() == 1, "Unsupported services amount: %s",
                info.getServiceCount());
        String payloadJson = info.getService(0).getServiceInfo().getPayload().getValue();
        ServicePayload payload = PayloadMapper.fromJson(payloadJson, payloadType);
        BookingPriceInfo priceInfo = payload.getActualCosts();

        order.setVariantId(payload.getVariantId());
        order.setTravellers(payload.getTravellers().stream()
                .map(t -> {
                    PassengerPricePart ppp = priceInfo != null ?
                            priceInfo.getPassengerPrices().get(t.getTravellerInfoId()) : null;
                    TravellerInfoDTO.TravellerInfoDTOBuilder builder = TravellerInfoDTO.builder()
                            .id(t.getTravellerInfoId())
                            .firstName(t.getFirstName())
                            .middleName(nonameToEmpty(t.getMiddleName()))
                            .lastName(t.getLastName())
                            .dateOfBirth(t.getDateOfBirth())
                            .documentNumber(t.getDocumentNumber())
                            .documentValidTill(t.getDocumentValidTill())
                            .documentType(t.getDocumentType())
                            .sex(t.getSex())
                            .category(t.getCategory())
                            .loyaltyProgramAccountNumber(t.getLoyaltyProgramAccountNumber())
                            .priceInfo(ppp != null ? convertPricePart(ppp) : null);
                    if (!Strings.isNullOrEmpty(t.getLoyaltyProgramAccountNumber())) {
                        builder.loyaltyProgramInternalCode(
                                airlineLoyaltyProgramCodeMapper.getInternalCodeByAirlineCode(t.getLoyaltyProgramCode())
                        );
                    }
                    return builder.build();
                })
                .collect(toList()));

        order.setPartner(payload.getPartnerId());
        order.setPreliminaryPrice(payload.getPreliminaryCost());
        if (priceInfo != null) {
            order.setPrice(priceInfo.getTotalAmount());
            //order.setAgencyMarkup(priceInfo.getAgencyMarkup());
        }

        Preconditions.checkArgument(info.getServiceCount() == 1,
                "Exactly 1 service is expected but have got %s", info.getServiceCount());
        if (SERVICE_ITEM_STATE_TO_ERROR_CODE.containsKey(payload.getBookingFailureReason())) {
            order.setErrorCode(SERVICE_ITEM_STATE_TO_ERROR_CODE.get(payload.getBookingFailureReason()));
        }

        Preconditions.checkArgument(info.getOrderType() == EOrderType.OT_AVIA_AEROFLOT,
                "Unexpected order type: %s", info.getOrderType());
        order.setAirReservation(convertAeroflotAirReservation((AeroflotServicePayload) payload));
        order.setEDisplayOrderState(info.getDisplayOrderState());
        order.setServicedAt(ProtoUtils.toLocalDateTime(info.getServicedAt()));
        order.setReference(referenceJsonFactory.createReferenceNode(order));

        order.setPromoCampaigns(aviaVariantDTOFactory.mapPromoCampaigns(payload.getPromoCampaignsInfo()));

        return order;
    }

    private String nonameToEmpty(String middleName) {
        if (NONAME_MIDDLENAME.equals(middleName)){
            return "";
        }
        return middleName;
    }

    private AirReservationDTO convertAeroflotAirReservation(AeroflotServicePayload payload) {
        AeroflotVariant variant = payload.getVariant();
        Map<String, List<AeroflotSegment>> segments = new LinkedHashMap<>();
        for (AeroflotSegment segment : variant.getSegments()) {
            segments.computeIfAbsent(segment.getOriginDestinationId(), key -> new ArrayList<>()).add(segment);
        }
        Variant genericVariant = AeroflotVariantConverter.convertVariant(variant);
        Map<String, SegmentFare> segmentsTermValues = payload.getFareTerms();
        if (segmentsTermValues == null) {
            String lang = variant.getContext().getLanguage();
            segmentsTermValues = fareFamilyService.getFareTerms(genericVariant, lang);
        }
        Map<String, SegmentFare> finalTerms = segmentsTermValues;
        AirReservationDTO airReservation = new AirReservationDTO();
        airReservation.setSegments(segments.values().stream()
                .map(segList -> SegmentDTO.builder()
                        .flights(segList.stream()
                                .map(seg -> FlightDTO.builder()
                                        .id(seg.getId())
                                        .departure(seg.getDeparture().getDateTime())
                                        .arrival(seg.getArrival().getDateTime())
                                        .from(airportCacheService.getByIataCode(seg.getDeparture().getAirportCode()).getId())
                                        .to(airportCacheService.getByIataCode(seg.getArrival().getAirportCode()).getId())
                                        .number(seg.getMarketingCarrier().getFlightNumber())
                                        .marketingAviaCompany(airlineCacheService.getByIataCode(seg.getMarketingCarrier().getAirlineId()).getId())
                                        .operatingAviaCompany(airlineCacheService.getByIataCode(seg.getOperatingCarrier().getAirlineId()).getId())
                                        .fareTerms(FareTermsHelper.fillFakeMilesProperties(
                                                Preconditions.checkNotNull(finalTerms.get(seg.getId()),
                                                        "No fare term for the segment id %s", seg.getId())))
                                        .build())
                                .collect(toList()))
                        .build())
                .collect(toList()));
        AeroflotOrderRef bookingRef = payload.getBookingRef();
        if (bookingRef != null) {
            airReservation.setPnr(bookingRef.getPnr());
        }
        return airReservation;
    }
}
