package ru.yandex.travel.api.services.avia.orders;

import java.time.Clock;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import javax.validation.Valid;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.ff.model.SegmentFare;
import ru.yandex.avia.booking.partners.gateways.BookingGateway;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AviaDicts;
import ru.yandex.avia.booking.partners.gateways.model.booking.ClientInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayloadInitParams;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.avia.booking.promo.AviaPromoCampaignsInfo;
import ru.yandex.avia.booking.service.dto.AeroflotStateDTO;
import ru.yandex.avia.booking.service.dto.CategoryPriceDTO;
import ru.yandex.avia.booking.service.dto.CompositeOrderStateDTO;
import ru.yandex.avia.booking.service.dto.FlightDTO;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.avia.booking.service.dto.OrderListDTO;
import ru.yandex.avia.booking.service.dto.VariantCheckToken;
import ru.yandex.avia.booking.service.dto.VariantDTO;
import ru.yandex.avia.booking.service.dto.form.CreateOrderForm;
import ru.yandex.avia.booking.service.dto.form.InitOrderPaymentForm;
import ru.yandex.avia.booking.service.dto.form.TravellerFormDTO;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.services.avia.AviaBookingProviderResolver;
import ru.yandex.travel.api.services.avia.references.AirlineLoyaltyProgramCodeMapper;
import ru.yandex.travel.api.services.avia.references.AviaGeobaseCountryService;
import ru.yandex.travel.api.services.avia.td.AviaTdInfo;
import ru.yandex.travel.api.services.avia.td.AviaTdInfoExtractor;
import ru.yandex.travel.api.services.avia.td.AviaTdSegment;
import ru.yandex.travel.api.services.avia.variants.AviaBookingProperties;
import ru.yandex.travel.api.services.avia.variants.AviaVariantService;
import ru.yandex.travel.api.services.common.PhoneCountryCodesService;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirlineDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaSettlementDictionary;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * The class will replace the current order service when the migration to the orchestrator is complete
 * and no old/new switch is needed anymore.
 */
@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
@Slf4j
public class AviaOrderService {
    public static final List<PassengerCategory> EXPECTED_PASSENGER_CATEGORIES_ORDER =
            List.of(PassengerCategory.ADULT, PassengerCategory.CHILD, PassengerCategory.INFANT);

    private final AviaBookingProperties properties;
    private final AviaOrchestratorClientAdapter orchestratorClient;
    private final AviaTdInfoExtractor ticketDaemonInfoExtractor;
    private final AviaBookingProviderResolver bookingGatewayResolver;
    private final AviaGeobaseCountryService countryMapper;
    private final AviaVariantService variantService;
    private final PhoneCountryCodesService phoneCountryCodesService;
    private final AirlineLoyaltyProgramCodeMapper airlineLoyaltyProgramCodeMapper;

    private final AviaAirlineDictionary aviaAirlineDictionary;
    private final AviaAirportDictionary aviaAirportDictionary;
    private final AviaSettlementDictionary aviaSettlementDictionary;

    private final Clock clock = Clock.systemUTC();
    static final int RU_COUNTRY_ID = 225;
    static final String NONAME_MIDDLENAME = "Noname";

    static void handleTravellersWithSameName(List<TravellerInfo> travellers) {
        Set<List<String>> uniqueNames = new HashSet<>();
        for (TravellerInfo traveller : travellers) {
            Preconditions.checkArgument(Strings.isNullOrEmpty(traveller.getLastNameSuffix()),
                    "Last name suffixes aren't supported");
            List<String> nameParts = new ArrayList<>();
            nameParts.add(normalize(traveller.getFirstName()));
            nameParts.add(normalize(traveller.getMiddleName()));
            nameParts.add(normalize(traveller.getLastName()));
            if (!uniqueNames.contains(nameParts)) {
                uniqueNames.add(nameParts);
            } else {
                // https://jira.aeroflot.ru/browse/NDCYAN-12
                // try with a suffix
                traveller.setLastNameSuffix("Jr");
                nameParts.add(traveller.getLastNameSuffix());
                Preconditions.checkArgument(!uniqueNames.contains(nameParts),
                        "Too many passengers with the same name");
                uniqueNames.add(nameParts);
            }
        }
    }

    private static String normalize(String s) {
        return s == null ? null : s.toUpperCase();
    }

    public CompletableFuture<OrderDTO> createOrder(@Valid CreateOrderForm createOrderForm,
                                                   UserCredentials userCredentials) {
        try {
            Preconditions.checkState(properties.getNewOrdersEnabled() == Boolean.TRUE, "Avia booking is disabled");
            return createOrderImpl(createOrderForm, userCredentials);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<OrderDTO> createOrderImpl(@Valid CreateOrderForm createOrderForm,
                                                        UserCredentials userCredentials) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(createOrderForm.getUserIp()),
                "No mandatory userIp parameter");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(createOrderForm.getUserAgent()),
                "No mandatory userAgent parameter");

        VariantCheckToken checkToken = VariantCheckToken.fromRaw(createOrderForm.getVariantToken());
        JsonNode checkData = variantService.getVariantInfo(checkToken.getAvailabilityCheckId()).deepCopy();

        if (!Strings.isNullOrEmpty(checkToken.getOfferId())) {
            setSelectedOffer(checkData, checkToken.getOfferId());
        }

        AviaTdInfo tdaemonInfo = ticketDaemonInfoExtractor.parseTdaemonInfo(checkData);

        BookingGateway gateway = bookingGatewayResolver.gatewayForPartner(
                tdaemonInfo.getPartnerCode(), tdaemonInfo.getTestContext());
        ParsedPhone parsedPhone = ParsedPhone.create(createOrderForm.getPhone(), phoneCountryCodesService);
        ClientInfo clientInfo = ClientInfo.builder()
                .email(createOrderForm.getEmail())
                .phone(createOrderForm.getPhone())
                .phoneCountryCode(parsedPhone.getCountryDialingCode())
                .phoneNumber(parsedPhone.getNumber())
                .userIp(createOrderForm.getUserIp())
                .userAgent(createOrderForm.getUserAgent())
                .build();

        VariantDTO variant = variantService.parseVariant(checkData);
        try {
            validateVariant(variant, clock);
        } catch (Exception e) {
            log.info("Received variant validation has failed; availability check '{}', variant id '{}'",
                    checkToken, variant.getId(), e);
            return CompletableFuture.failedFuture(e);
        }

        List<PassengerCategory> expectedCategories = getExpectedCategories(
                variant.getVariantPriceInfo().getCategoryPrices());
        AeroflotVariant aeroflotVariant = (AeroflotVariant) gateway.resolveVariantInfo(checkData);
        List<TravellerInfo> travellers = convertTravellers(
                createOrderForm.getDocuments(),
                tdaemonInfo,
                countryMapper,
                airlineLoyaltyProgramCodeMapper,
                expectedCategories,
                hasRussiaSegment(aeroflotVariant));

        Map<String, SegmentFare> segmentsTermValues = variant.getLegs().stream()
                .flatMap(leg -> leg.getFlights().stream())
                .collect(toMap(FlightDTO::getId, FlightDTO::getFareTerms));
        String offerId = variant.getVariantPriceInfo().getId();
        AviaPromoCampaignsInfo promoCampaignsInfo = variantService.getPromoCampaignsInfo(tdaemonInfo, offerId);
        ServicePayload payload = gateway.createServicePayload(ServicePayloadInitParams.builder()
                .variantId(checkToken.toString())
                .variantToken(aeroflotVariant)
                // the user should have agreed on the potentially changed price to reach this point,
                // making the actual price a new baseline
                .preliminaryPrice(variant.getVariantPriceInfo().getTotal())
                .clientInfo(clientInfo)
                .travellers(travellers)
                .fareTerms(segmentsTermValues)
                .promoCampaignsInfo(promoCampaignsInfo)
                .build());

        fillAviaDicts((AeroflotServicePayload) payload);
        return orchestratorClient.createOrder(createOrderForm, payload, userCredentials, tdaemonInfo.getTestContext());
    }

    public CompletableFuture<ArrayList<AeroflotStateDTO>> getAeroflotState(List<UUID> orderIds) {
        try {
            Preconditions.checkArgument(orderIds != null && !orderIds.isEmpty(),
                    "No mandatory or empty orderIds parameter");
            return orchestratorClient.getAeroflotState(orderIds);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<AeroflotStateDTO> getAeroflotState(UUID orderId) {
        try {
            Preconditions.checkArgument(orderId != null, "orderId must be non-null");
            return orchestratorClient.getAeroflotState(List.of(orderId)).thenApply(states -> states.get(0));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<AeroflotStateDTO> getAeroflotStateWithTimeout(UUID orderId, long timeout, TimeUnit unit) {
        try {
            Preconditions.checkArgument(orderId != null, "orderId must be non-null");
            return orchestratorClient.getAeroflotState(List.of(orderId)).thenApply(states -> states.get(0)).completeOnTimeout(null, timeout, unit);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private boolean hasRussiaSegment(AeroflotVariant variant) {
        return variant.getSegments().stream().anyMatch(
                seg -> isRussiaSegment(seg.getDeparture().getAirportCode(), seg.getArrival().getAirportCode()));
    }

    private boolean isRussiaSegment(String departureAirportCode, String arrivalAirportCode) {
        return isRussiaCode(departureAirportCode) && isRussiaCode(departureAirportCode);
    }

    private boolean isRussiaCode(String airportCode) {
        if (!aviaAirportDictionary.hasIataCode(airportCode)) {
            return false;
        }
        long settlementId = aviaAirportDictionary.getByIataCode(airportCode).getSettlementId();
        if (!aviaSettlementDictionary.hasId(settlementId)) {
            return false;
        }
        return aviaSettlementDictionary.getById(settlementId).getCountryId() == RU_COUNTRY_ID;
    }

    private void fillAviaDicts(AeroflotServicePayload payload) {
        List<String> pointCodes = variantPoints(payload.getVariant());
        payload.setAviaDicts(
                AviaDicts.builder()
                        .carriers(variantCarriers(payload.getVariant()))
                        .stations(variantStations(pointCodes))
                        .settlements(variantSettlements(pointCodes))
                        .build()
        );
    }

    private List<String> variantPoints(AeroflotVariant variant) {
        var segments = variant.getSegments().stream()
                .flatMap(seg -> Stream.of(seg.getDeparture().getAirportCode(), seg.getArrival().getAirportCode()))
                .filter(code -> !Strings.isNullOrEmpty(code))
                .distinct();

        var originDestinations = variant.getOriginDestinations().stream()
                .flatMap(od -> Stream.of(od.getDepartureCode(), od.getArrivalCode()))
                .filter(code -> !Strings.isNullOrEmpty(code))
                .distinct();

        return Stream.concat(segments, originDestinations).distinct().collect(toList());
    }

    private List<AviaDicts.Station> variantStations(List<String> variantPoints) {
        return variantPoints.stream()
                .filter(aviaAirportDictionary::hasIataCode)
                .map(code -> AviaDicts.Station.builder()
                        .code(code)
                        .raspID(aviaAirportDictionary.getByIataCode(code).getId())
                        .build()
                )
                .collect(toList());
    }


    private List<AviaDicts.Settlement> variantSettlements(List<String> variantPoints) {
        return variantPoints.stream()
                .filter(aviaSettlementDictionary::hasCode)
                .map(code -> AviaDicts.Settlement.builder()
                        .code(code)
                        .raspID(aviaSettlementDictionary.getByCode(code).getId())
                        .geoID(aviaSettlementDictionary.getByCode(code).getGeoId())
                        .build()
                )
                .collect(toList());
    }

    private List<AviaDicts.Carrier> variantCarriers(AeroflotVariant variant) {
        return variant.getSegments().stream()
                .flatMap(seg ->
                        Stream.of(seg.getMarketingCarrier().getAirlineId(), seg.getOperatingCarrier().getAirlineId())
                )
                .filter(code -> !Strings.isNullOrEmpty(code))
                .distinct()
                .filter(aviaAirlineDictionary::hasIataCode)
                .map(code ->
                        AviaDicts.Carrier.builder()
                                .code(code)
                                .raspID(aviaAirlineDictionary.getByIataCode(code).getId())
                                .build()
                )
                .collect(toList());
    }

    static void validateVariant(VariantDTO variant, Clock clock) {
        LocalDateTime departure = variant.getLegs().stream()
                .flatMap(leg -> leg.getFlights().stream())
                .map(FlightDTO::getDeparture)
                .min(Comparator.naturalOrder())
                .orElse(null);
        if (departure == null) {
            throw new IllegalArgumentException("No departure date for variant " + variant.getId());
        }
        // a not so strict check: only today with regards to UTC-0..12 time zones
        LocalDateTime minAllowedDeparture = LocalDateTime.now(clock).minusHours(12);
        if (departure.isBefore(minAllowedDeparture)) {
            throw new IllegalArgumentException("Departure date in the past: " + departure + "; " +
                    "variant id '" + variant.getId() + "'");
        }
    }

    private List<PassengerCategory> getExpectedCategories(List<CategoryPriceDTO> categoryPrices) {
        List<PassengerCategory> categories = new ArrayList<>();
        for (CategoryPriceDTO categoryPrice : categoryPrices) {
            for (int i = 0; i < categoryPrice.getQuantity(); i++) {
                categories.add(categoryPrice.getPassengerCategory());
            }
        }
        categories.sort(Comparator.comparing(EXPECTED_PASSENGER_CATEGORIES_ORDER::indexOf));
        return categories;
    }

    public CompletableFuture<OrderDTO> getOrder(UUID orderId) {
        return orchestratorClient.getOrderInfo(orderId);
    }

    public CompletableFuture<OrderDTO> getOrder(UUID orderId, boolean refreshStatus) {
        if (!refreshStatus || !properties.isEnableOrderStateRefreshing()) {
            return getOrder(orderId);
        }
        var getOrderFuture = orchestratorClient.getOrderInfo(orderId);
        var getStateFuture = getAeroflotStateWithTimeout(orderId, properties.getRefreshStateTimeout().toSeconds(), TimeUnit.SECONDS);
        return CompletableFuture.allOf(getOrderFuture, getStateFuture).thenApply(__ -> {
            var order = getOrderFuture.join();
            var state = getStateFuture.join();
            if (state != null) {
                order.setState(AviaOrchestratorEnumsMapper.lookup(
                        AviaOrchestratorEnumsMapper.ORDER_STATE_PROTO_TO_API, state.getState(), order.getState()));
                order.setEDisplayOrderState(AviaOrchestratorEnumsMapper.lookup(
                        AviaOrchestratorEnumsMapper.AEROFLOT_ORDER_TO_EDISPLAY_ORDER_STATE, state.getState(), order.getEDisplayOrderState()));
                order.setStateRefreshed(true);
            }
            return order;
        });
    }

    public CompletableFuture<CompositeOrderStateDTO> getCompositeOrderState(UUID id) {
        return orchestratorClient.getState(id, properties.getPreferSlaveForStatus());
    }

    public CompletableFuture<Void> startPaymentAsync(UUID orderId, InitOrderPaymentForm params) {
        return orchestratorClient.startPayment(orderId, params);
    }

    static List<TravellerInfo> convertTravellers(List<TravellerFormDTO> documents, AviaTdInfo tdaemonInfo,
                                                 AviaGeobaseCountryService countryMapper,
                                                 AirlineLoyaltyProgramCodeMapper airlineLoyaltyProgramCodeMapper,
                                                 List<PassengerCategory> expectedCategories,
                                                 boolean isRussiaOnlyVariant) {
        Preconditions.checkArgument(documents.size() == expectedCategories.size(),
                "Unexpected number of documents: expected %s but got %s", expectedCategories.size(), documents.size());
        LocalDateTime lastFlightDate = getLastFlightArrival(tdaemonInfo);
        int nextId = 1;
        List<TravellerInfo> travellers = new ArrayList<>();
        for (TravellerFormDTO t : documents) {
            Preconditions.checkArgument(!Strings.isNullOrEmpty(t.getCitizenship()) || t.getCitizenshipGeoId() != null,
                    "At least one of the traveller fields should be specified: citizenship, citizenshipGeoId");
            PassengerCategory expectedCategory = expectedCategories.get(nextId - 1);
            TravellerInfo result = new TravellerInfo();
            result.setTravellerInfoId(String.valueOf(nextId++));
            result.setFirstName(trimNameWhitespaces(t.getFirstName()));
            // RASPTICKETS-21621: temporary hack to let orders pass Aeroflot validations
            result.setMiddleName(nonameOrEmpty(trimNameWhitespaces(t.getMiddleName()), isRussiaOnlyVariant));
            result.setLastName(trimNameWhitespaces(t.getLastName()));
            result.setDateOfBirth(t.getDateOfBirth());
            result.setDocumentNumber(t.getDocumentNumber());
            result.setDocumentValidTill(t.getDocumentValidTill());
            result.setDocumentType(t.getDocumentType());
            result.setSex(t.getSex());
            // waiting for RASPTICKETS-13783 resolution
            result.setNationalityCode(!Strings.isNullOrEmpty(t.getCitizenship()) ?
                    t.getCitizenship() :
                    countryMapper.getIsoName(t.getCitizenshipGeoId()));
            result.setNationalityCodeGeoId(t.getCitizenshipGeoId() != null ?
                    t.getCitizenshipGeoId() :
                    countryMapper.getCountryGeoId(t.getCitizenship()));
            result.setCategory(ensureCategoryByBirthDate(
                    expectedCategory, t.getDateOfBirth(), lastFlightDate.toLocalDate()));

            if (!Strings.isNullOrEmpty(t.getLoyaltyProgramInternalCode())) {
                result.setLoyaltyProgramCode(
                        airlineLoyaltyProgramCodeMapper.getAirlineCodeByInternalCode(t.getLoyaltyProgramInternalCode())
                );
            }
            result.setLoyaltyProgramAccountNumber(t.getLoyaltyProgramAccountNumber());
            travellers.add(result);
        }
        handleTravellersWithSameName(travellers);
        return travellers;
    }

    static String nonameOrEmpty(String middleName, boolean isRussiaOnlyVariant) {
        if (isRussiaOnlyVariant && Strings.isNullOrEmpty(middleName)) {
            return NONAME_MIDDLENAME;
        }
        return middleName;
    }

    private static String trimNameWhitespaces(String name) {
        return name == null ? null : name.trim().replaceAll(" +", " ");
    }

    private static PassengerCategory ensureCategoryByBirthDate(PassengerCategory category,
                                                               LocalDate birthDate, LocalDate momentToCompare) {
        long age = ChronoUnit.YEARS.between(birthDate, momentToCompare);
        if (age < 2) {
            Preconditions.checkArgument(category == PassengerCategory.INFANT || category == PassengerCategory.CHILD,
                    "Expected one the INFANT/CHILD categories for the passenger of age %s but got %s",
                    age, category);
        } else if (age < 12) {
            Preconditions.checkArgument(category == PassengerCategory.CHILD,
                    "Expected the CHILD category for the passenger of age %s but got %s", age, category);
        } else {
            Preconditions.checkArgument(category == PassengerCategory.ADULT,
                    "Expected the ADULT category for the passenger of age %s but got %s", age, category);
        }
        return category;
    }

    private void setSelectedOffer(JsonNode variantInfo, String offerId) {
        JsonNode priceNode = variantInfo.path("price_info");
        JsonNode variantNode = variantInfo.path("variant_info").path("variant");
        Preconditions.checkArgument(priceNode instanceof ObjectNode,
                "invalid variant price info: offerId=%s, node=%s", offerId, priceNode);
        Preconditions.checkArgument(variantInfo instanceof ObjectNode,
                "invalid variant info: offerId=%s, node=%s", offerId, variantNode);

        JsonNode offers = variantInfo.path("all_variants");
        JsonNode selectedOffer = null;
        for (JsonNode offer : offers) {
            String id = offer.path("variantPriceInfo").path("id").asText();
            Preconditions.checkArgument(!Strings.isNullOrEmpty(id),
                    "illegal offer format: requested=%s, offer=%s", offerId, offer);
            if (id.equals(offerId)) {
                selectedOffer = offer;
                break;
            }
        }
        Preconditions.checkNotNull(selectedOffer, "offer not found: id=%s", offerId);

        JsonNode selectedPrice = selectedOffer.path("variantPriceInfo").path("total").deepCopy();
        ((ObjectNode) priceNode).set("first_check_price", selectedPrice);
        ((ObjectNode) variantNode).set("legs", selectedOffer.get("legs"));
        ((ObjectNode) variantNode).set("variantPriceInfo", selectedOffer.get("variantPriceInfo"));
        ((ObjectNode) variantInfo.at("/order_data/booking_info")).put("OfferId", offerId);
        ((ObjectNode) variantInfo).remove("all_variants");
    }

    private static LocalDateTime getLastFlightArrival(AviaTdInfo tdaemonInfo) {
        List<AviaTdSegment> tdSegments = tdaemonInfo.getSegments();
        int segmentIdx = tdSegments.size() - 1;
        int flightIdx = tdSegments.get(segmentIdx).getSegments().size() - 1;
        return tdSegments.get(segmentIdx).getSegments().get(flightIdx).getArrivalDateTime();
    }

    public CompletableFuture<OrderListDTO> listOrders(int page, int pageSize, List<EDisplayOrderState> states) {
        return orchestratorClient.listOrders(page, pageSize, states);
    }

    // todo(tlg-13): TRAVELBACK-1149: should use unification service api instead, will be removed before the release
    @Value
    static class ParsedPhone {
        private final int countryDialingCode;
        private final long number;

        public static ParsedPhone create(String phone, PhoneCountryCodesService phoneCountryCodesService) {
            String normalizedPhone = phone.replaceAll("[^0-9]+", "").replaceAll("^0+", "");
            if (normalizedPhone.length() == 11 && normalizedPhone.startsWith("8")) {
                normalizedPhone = "7" + normalizedPhone.substring(1);
            }
            if (normalizedPhone.length() == 10 && (normalizedPhone.startsWith("9") || normalizedPhone.startsWith("4"))) {
                normalizedPhone = "7" + normalizedPhone;
            }
            Preconditions.checkState(normalizedPhone.length() > 1, "Illegal phone number: [%s]", phone);
            try {
                Integer countryCode = phoneCountryCodesService.getCountryCodeSafe(normalizedPhone);
                int countryCodeLength = String.valueOf(countryCode).length();
                long number = Long.parseLong(normalizedPhone.substring(countryCodeLength));
                return new ParsedPhone(countryCode, number);
            } catch (Exception e) {
                log.info("Phone {} country code parsing failed", phone, e);
                return fallback2(phone);
            }
        }

        public static ParsedPhone fallback2(String phone) {
            int countryCode = Integer.parseInt(phone.substring(0, 1));
            long number = Long.parseLong(phone.substring(1));
            return new ParsedPhone(countryCode, number);
        }
    }
}
