package ru.yandex.travel.api.services.avia.references;

import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.geobase6.RegionHash;
import ru.yandex.geobase6.Value;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;

@Slf4j
public class AviaGeobaseCountryServiceImpl implements AviaGeobaseCountryService {
    private static final int COUNTRY_REGION_TYPE = 3;

    private final GeoBase geoBase;
    private final Map<String, Integer> isoName2Id;

    public AviaGeobaseCountryServiceImpl(GeoBase geoBase) {
        this.geoBase = geoBase;
        this.isoName2Id = ImmutableMap.copyOf(collectCountryCodesMapping());
    }

    @Override
    public String getIsoName(Integer regionId) {
        RegionHash region = geoBase.getRegionById(regionId);
        Value attr = region.getAttr("iso_name");
        if (attr == null || Strings.isNullOrEmpty(attr.getString())) {
            throw new RuntimeException("No iso code for region: " + regionId);
        }
        return attr.getString();
    }

    @Override
    public Integer getCountryGeoId(String countryIsoCode) {
        Integer regionId = isoName2Id.get(countryIsoCode);
        if (regionId == null) {
            throw new RuntimeException("Unknown iso code: " + countryIsoCode);
        }
        return regionId;
    }

    @Override
    public Map<String, Integer> collectCountryCodesMapping() {
        HashMap<String, Integer> mapping = new HashMap<>();
        for (int id : geoBase.getRegionsIdsByType(COUNTRY_REGION_TYPE)) {
            String isoCode = geoBase.getRegionById(id).getString("iso_name");
            Integer replacedId = mapping.put(isoCode, id);
            if (replacedId != null) {
                log.warn("Iso code '{}' collision for region ids: [{}, {}]", isoCode, replacedId, id);
            }
        }
        return mapping;
    }
}
