package ru.yandex.travel.api.services.avia.references;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.service.dto.FlightDTO;
import ru.yandex.avia.booking.service.dto.FlightStopDTO;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.avia.booking.service.dto.SegmentDTO;
import ru.yandex.avia.booking.service.dto.TravellerInfoDTO;
import ru.yandex.avia.booking.service.dto.VariantDTO;
import ru.yandex.avia.booking.service.dto.reference.AirlineReferenceDTO;
import ru.yandex.avia.booking.service.dto.reference.AirportReferenceDTO;
import ru.yandex.avia.booking.service.dto.reference.LoyaltyProgramInfoDTO;
import ru.yandex.avia.booking.service.dto.reference.ReferenceDTO;
import ru.yandex.avia.booking.service.dto.reference.SettlementReferenceDTO;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirlineDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaSettlementDictionary;

import static java.util.stream.Collectors.toList;

@Service
@RequiredArgsConstructor
public class AviaReferenceJsonFactory {
    private static final ReferenceDTO EMPTY_REFERENCE_DTO = new ReferenceDTO();

    private final AviaAirlineDictionary airlineDictionary;
    private final AviaAirportDictionary airportDictionary;
    private final AviaSettlementDictionary settlementDictionary;

    public ReferenceDTO createReferenceNode(OrderDTO orderDTO) {
        if (orderDTO.getAirReservation() != null) {
            List<String> loyaltyPrograms = orderDTO.getTravellers().stream().map(TravellerInfoDTO::getLoyaltyProgramInternalCode)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toUnmodifiableList());
            return collectReferenceDTO(orderDTO.getAirReservation().getSegments(), loyaltyPrograms);
        } else {
            return EMPTY_REFERENCE_DTO;
        }
    }

    public ReferenceDTO createReferenceNode(VariantDTO variantDTO) {
        return collectReferenceDTO(variantDTO.getLegs(), variantDTO.getAllowedLoyaltyPrograms());
    }

    private void collectInfoFromFlight(Set<Long> airportIds, Set<Long> airlineIds, FlightDTO flightDTO) {
        airportIds.add(flightDTO.getFrom());
        airportIds.add(flightDTO.getTo());
        airlineIds.add(flightDTO.getMarketingAviaCompany());
        airlineIds.add(flightDTO.getOperatingAviaCompany());
        if (flightDTO.getStops() != null) {
            for (FlightStopDTO flightStop : flightDTO.getStops()) {
                airportIds.add(flightStop.getAirportId());
            }
        }
    }

    private ReferenceDTO collectReferenceDTO(List<SegmentDTO> segments, List<String> loyaltyProgramCodes) {
        Set<Long> airportIds = new HashSet<>();
        Set<Long> airlineIds = new HashSet<>();
        for (SegmentDTO segmentDTO : segments) {
            for (FlightDTO flightDTO : segmentDTO.getFlights()) {
                collectInfoFromFlight(airportIds, airlineIds, flightDTO);
            }
        }
        ReferenceDTO result = new ReferenceDTO();
        result.setLoyaltyPrograms(createLoyaltyProgramInfos(loyaltyProgramCodes));
        result.setAirlines(createAirlines(airlineIds));
        List<AirportReferenceDTO> airports = createAirports(airportIds);
        List<Long> settlementIds = airports.stream().map(AirportReferenceDTO::getSettlement).collect(toList());
        result.setAirports(airports);
        result.setSettlements(createSettlements(settlementIds));
        return result;
    }

    private List<LoyaltyProgramInfoDTO> createLoyaltyProgramInfos(List<String> loyaltyProgramCodes) {
        return loyaltyProgramCodes.stream().map(internalCode -> {
            AirlineLoyaltyProgramType programType =
                    AirlineLoyaltyProgramType.BY_INTERNAL_CODE.getByValue(internalCode);
            LoyaltyProgramInfoDTO r = new LoyaltyProgramInfoDTO();
            r.setName(programType.getTitle());
            r.setInternalCode(internalCode);
            return r;
        }).collect(Collectors.toUnmodifiableList());
    }

    private List<AirlineReferenceDTO> createAirlines(Set<Long> airlines) {
        return airlines.stream()
                .map(airlineDictionary::getById)
                .map(AviaReferenceDTOMapper::createAirlineReferenceDTO)
                .collect(toList());
    }

    private List<AirportReferenceDTO> createAirports(Set<Long> airports) {
        return airports.stream()
                .map(airportDictionary::getById)
                .map(AviaReferenceDTOMapper::createAirportReferenceDTO)
                .collect(toList());
    }

    private List<SettlementReferenceDTO> createSettlements(List<Long> settlementIds) {
        return settlementIds.stream()
                .map(settlementDictionary::getById)
                .map(AviaReferenceDTOMapper::createSettlementReferenceDTO)
                .collect(toList());
    }
}
