package ru.yandex.travel.api.services.avia.td;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonUtils;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.services.avia.td.promo.AviaTdAeroflotPlus2021Offer;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;

@Service
@RequiredArgsConstructor
@Slf4j
public class AviaTdInfoExtractor {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss");

    private final ApiTokenEncrypter tokenEncrypter;

    public AviaTdInfo parseTdaemonInfo(JsonNode jsonNode) {
        JsonNode partnerNode = jsonNode.path("order_data").path("partner");
        if (partnerNode.isMissingNode()) {
            throw new RuntimeException("Partner node is missing in json acquired by token");
        }
        String partnerCode = partnerNode.asText();
        JsonNode tariff = jsonNode.path("variant").path("tariff");
        if (tariff.isMissingNode()) {
            throw new RuntimeException("Variant node is missing in json acquired by token");
        }
        Money preliminaryPrice = parseTotalCost(jsonNode.path("variant"));

        String variantTag = jsonNode.at("/variant/tag").textValue();
        String qid = jsonNode.at("/order_data/qid").textValue();
        String qkey = jsonNode.path("order_data").path("qkey").textValue();
        String[] queryParts = qkey.split("_");

        AviaTdInfo result = new AviaTdInfo();
        result.setVariantQid(qid);
        result.setVariantTag(variantTag);
        result.setPartnerCode(partnerCode);
        result.setRawData(jsonNode);
        result.setPreliminaryPrice(preliminaryPrice);
        result.setSegments(parseTdSegments(jsonNode));
        result.setAdultCount(Integer.parseInt(queryParts[5]));
        result.setChildrenCount(Integer.parseInt(queryParts[6]));
        result.setInfantCount(Integer.parseInt(queryParts[7]));
        result.setSearchParams(AviaTicketDaemonUtils.parseSearchParamsSafe(jsonNode));

        String testContextToken = jsonNode.path("variantTestContext").textValue();
        if (!Strings.isNullOrEmpty(testContextToken)) {
            TAviaTestContext testContext = tokenEncrypter.fromAviaTestContextToken(testContextToken);
            result.setTestContext(testContext);
        }

        result.setPromoCampaigns(parsePromoCampaigns(jsonNode));

        return result;
    }

    private Collection<String> parsePromo2020Ids(JsonNode jsonNode) {
        JsonNode promoIdsNode = jsonNode.at("/order_data/booking_info/PromoOfferIds");
        if (promoIdsNode != null && promoIdsNode.isArray()) {
            Collection<String> promoIds = new ArrayList<>();
            for (JsonNode idValue : promoIdsNode) {
                promoIds.add(idValue.textValue());
            }
            return promoIds;
        }
        return null;
    }

    public List<AviaTdSegment> parseTdSegments(JsonNode tdInfoNode) {
        Map<String, AviaTdFlight> parsedSegments = new HashMap<>();
        JsonNode segmentNodes = tdInfoNode.get("flights");
        for (Iterator<Map.Entry<String, JsonNode>> it = segmentNodes.fields(); it.hasNext(); ) {
            Map.Entry<String, JsonNode> fieldEntry = it.next();
            parsedSegments.put(fieldEntry.getKey(), parseBookingFlight(fieldEntry.getValue()));
        }
        JsonNode variantNode = tdInfoNode.get("variant");
        JsonNode routeNode = variantNode.get("route");
        List<AviaTdSegment> legs = new ArrayList<>();
        for (JsonNode segmentNode : routeNode) {
            if (segmentNode.isArray() && segmentNode.size() > 0) {
                List<AviaTdFlight> mappedSegments = new ArrayList<>();
                for (JsonNode flightRefNode : segmentNode) {
                    AviaTdFlight flight = parsedSegments.get(flightRefNode.textValue());
                    if (flight != null) {
                        mappedSegments.add(flight);
                    } else {
                        throw new RuntimeException(String.format(
                                "Couldn't find flight with key %s", flightRefNode.textValue())
                        );
                    }
                }
                legs.add(
                        AviaTdSegment.builder()
                                .segments(mappedSegments)
                                .build()
                );
            }
        }
        //TODO hack in hope of unique segments
        int idx = 0;
        for (AviaTdSegment leg : legs) {
            for (AviaTdFlight segment : leg.getSegments()) {
                segment.setId("SEG_" + idx);
                idx++;
            }
        }
        return legs;
    }

    public Money parseTotalCost(JsonNode variantNode) {
        String currencyCode = variantNode.get("tariff").get("currency").textValue();
        if ("RUR".equalsIgnoreCase(currencyCode)) {
            currencyCode = "RUB";
        }
        return Money.of(
                variantNode.get("tariff").get("value").doubleValue(),
                currencyCode
        );
    }

    private AviaTdFlight parseBookingFlight(JsonNode flightNode) {
        flightNode.get("to").textValue();
        return AviaTdFlight.builder()
                .departureAirport(flightNode.get("from").asLong())
                .arrivalAirport(flightNode.get("to").asLong())
                .flightNumber(parseFlightNumber(flightNode))
                .departureDateTime(parseFlightDateTime(flightNode.get("departure")))
                .arrivalDateTime(parseFlightDateTime(flightNode.get("arrival")))
                .marketingAirlineCode(flightNode.get("company").asLong())
                .operatingAirlineCode(flightNode.get("company").asLong())
                .build();
    }

    private String parseFlightNumber(JsonNode flightNode) {
        return flightNode.get("number").textValue().split(" ")[1];
    }

    private LocalDateTime parseFlightDateTime(JsonNode flightDateTime) {
        return LocalDateTime.parse(flightDateTime.get("local").textValue(), DATE_TIME_FORMATTER);
    }

    private AviaTdPromoCampaigns parsePromoCampaigns(JsonNode tdDataJson) {
        try {
            return AviaTdPromoCampaigns.builder()
                    .promo2020Ids(parsePromo2020Ids(tdDataJson))
                    .aeroflotPlusPromo2021Offers(parseAeroflotPlus2021PromoOffers(tdDataJson))
                    .build();
        } catch (Exception e) {
            log.warn("Promo campaigns parsing has failed, skipping them", e);
            return AviaTdPromoCampaigns.builder().build();
        }
    }

    private List<AviaTdAeroflotPlus2021Offer> parseAeroflotPlus2021PromoOffers(JsonNode tdDataJson) {
        JsonNode promoNode = tdDataJson.at("/order_data/booking_info/AeroflotPlusPromo2021");
        List<AviaTdAeroflotPlus2021Offer> offers = new ArrayList<>();
        for (JsonNode offerNode : promoNode) {
            List<AviaTdAeroflotPlus2021Offer.Code> codes = new ArrayList<>();
            for (JsonNode code : offerNode.get("PlusPoints")) {
                codes.add(new AviaTdAeroflotPlus2021Offer.Code(code.get("Points").intValue()));
            }
            if (codes.isEmpty()) {
                log.warn("Broken AeroflotPlus2021 offer: no codes provided, skipping it; data: {}", offerNode);
                continue;
            }
            offers.add(AviaTdAeroflotPlus2021Offer.builder()
                    .offerId(offerNode.get("OfferId").textValue())
                    .plusCodes(codes)
                    .build());
        }
        return offers;
    }
}
