package ru.yandex.travel.api.services.avia.variants;

import java.time.Instant;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.dao.CannotAcquireLockException;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.PessimisticLockingFailureException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.services.avia.variants.model.AviaCachedVariantCheck;
import ru.yandex.travel.api.services.avia.variants.repositories.AviaCachedVariantRepository;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
@Slf4j
class AviaVariantCacheService {
    private final AviaBookingProperties properties;
    private final AviaCachedVariantRepository cachedVariantRepository;

    @Transactional(propagation = Propagation.MANDATORY)
    AviaCachedVariantCheck findOrCreateCachedCheck(@NonNull String partnerId, @NonNull String variantId) throws AviaVariantCacheRecordLockException {
        String id = partnerId + "/" + variantId;
        log.debug("Trying to lock the cache record for the requested availability check: id={}", id);
        AviaCachedVariantCheck cachedVariantCheck;
        try {
            cachedVariantCheck = cachedVariantRepository.selectForUpdateNoWait(partnerId, variantId);
        } catch (CannotAcquireLockException e) {
            log.debug("Can't lock the cache record, need to retry later: id={}", id);
            throw new AviaVariantCacheRecordLockException("failed to lock the existing record: id=" + id, e);
        }
        if (cachedVariantCheck != null) {
            log.debug("The cache record has been found and locked: id={}, expires_at={}",
                    id, cachedVariantCheck.getExpiresAt());
            if (cachedVariantCheck.getExpiresAt() != null && cachedVariantCheck.getExpiresAt().isBefore(Instant.now())) {
                log.debug("The found cache record already expired, preparing it for a refresh: id={}", id);
                cachedVariantCheck.setExpiresAt(null);
                cachedVariantCheck.setCheckId(null);
                cachedVariantRepository.update(cachedVariantCheck);
            }
        } else {
            log.debug("No cached record has been found, creating a new one: id={}", id);
            cachedVariantCheck = new AviaCachedVariantCheck();
            cachedVariantCheck.setPartnerId(partnerId);
            cachedVariantCheck.setVariantId(variantId);
            try {
                cachedVariantCheck.setExpiresAt(nextExpiration());
                cachedVariantRepository.insert(cachedVariantCheck);
            } catch (DataIntegrityViolationException | PessimisticLockingFailureException e) {
                log.debug("Can't create a new cache record, need to retry later: id={}", id);
                throw new AviaVariantCacheRecordLockException("failed to create a unique cache record: id=" + id, e);
            }
        }
        return cachedVariantCheck;
    }

    @Transactional(propagation = Propagation.MANDATORY)
    public void updateWithNewExpiration(AviaCachedVariantCheck cacheRecord) {
        log.debug("Availability checks cache update: cacheRecord={}", cacheRecord);
        cacheRecord.setExpiresAt(nextExpiration());
        cachedVariantRepository.update(cacheRecord);
    }

    private Instant nextExpiration() {
        return Instant.now().plusSeconds(properties.getVariantCache().getCheckTtl().getSeconds());
    }
}
