package ru.yandex.travel.api.services.avia.variants;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.ff.FareTermsHelper;
import ru.yandex.avia.booking.ff.model.SegmentFare;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData;
import ru.yandex.avia.booking.partners.gateways.model.booking.CodeContext;
import ru.yandex.avia.booking.partners.gateways.model.search.CategoryPrice;
import ru.yandex.avia.booking.partners.gateways.model.search.FareInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Flight;
import ru.yandex.avia.booking.partners.gateways.model.search.FlightStop;
import ru.yandex.avia.booking.partners.gateways.model.search.PriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Segment;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.avia.booking.promo.AeroflotPlusPromoInfo;
import ru.yandex.avia.booking.promo.AviaPromoCampaignsInfo;
import ru.yandex.avia.booking.service.dto.BookingParametersDto;
import ru.yandex.avia.booking.service.dto.CategoryPriceDTO;
import ru.yandex.avia.booking.service.dto.FareInfoDTO;
import ru.yandex.avia.booking.service.dto.FlightDTO;
import ru.yandex.avia.booking.service.dto.FlightStopDTO;
import ru.yandex.avia.booking.service.dto.SearchDataDTO;
import ru.yandex.avia.booking.service.dto.SegmentDTO;
import ru.yandex.avia.booking.service.dto.VariantCheckToken;
import ru.yandex.avia.booking.service.dto.VariantDTO;
import ru.yandex.avia.booking.service.dto.VariantPriceInfoDTO;
import ru.yandex.avia.booking.service.dto.promo.AeroflotPlusPromo2021DTO;
import ru.yandex.avia.booking.service.dto.promo.AviaPromo2020DTO;
import ru.yandex.avia.booking.service.dto.promo.AviaPromoCampaignsDTO;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.services.avia.fares.AviaFareFamilyService;
import ru.yandex.travel.api.services.avia.references.AirlineLoyaltyProgramType;
import ru.yandex.travel.api.services.avia.references.AviaGeoDataService;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirlineDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.dicts.avia.TAirline;
import ru.yandex.travel.dicts.avia.TAirport;

import static java.util.stream.Collectors.toList;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
@Slf4j
public class AviaVariantDTOFactory {
    private final AviaAirportDictionary airportDictionary;
    private final AviaAirlineDictionary airlineDictionary;
    private final AviaFareFamilyService fareFamilyService;
    private final AviaGeoDataService geoDataService;

    public static final List<String> ALLOWED_LOYALTY_PROGRAM_CODES = Arrays.stream(AirlineLoyaltyProgramType.values())
            .map(AirlineLoyaltyProgramType::getInternalCode).collect(Collectors.toUnmodifiableList());

    public VariantDTO mapVariantDTO(VariantCheckToken checkId, Variant variant) {
        VariantDTO result = new VariantDTO();
        VariantPriceInfoDTO variantPriceInfo = mapPriceInfo(variant.getPriceInfo());

        Map<String, SegmentFare> fareTerms = fareFamilyService.getFareTerms(variant, variant.getLang());
        result.setId(checkId.toString());
        result.setSearchData(mapSearchData(variant.getSearchData()));
        result.setVariantPriceInfo(variantPriceInfo);
        result.setLegs(mapLegs(variant.getSegments(), fareTerms, variant.getCodeContext()));
        result.setPromoCampaigns(mapPromoCampaigns(variant.getPriceInfo().getPromoCampaigns()));
        result.setAllowedLoyaltyPrograms(ALLOWED_LOYALTY_PROGRAM_CODES);
        result.setBookingParameters(BookingParametersDto.builder().build());
        return result;
    }

    private SearchDataDTO mapSearchData(SearchData searchData) {
        return SearchDataDTO.builder()
                .qid(searchData.getQid())
                .externalBookingUrl(searchData.getExternalBookingUrl())
                .build();
    }

    private VariantPriceInfoDTO mapPriceInfo(PriceInfo priceInfo) {
        VariantPriceInfoDTO result = new VariantPriceInfoDTO();
        result.setId(priceInfo.getId());
        result.setTotal(priceInfo.getTotal());
        result.setCategoryPrices(priceInfo.getCategoryPrices().stream()
                .map(this::mapCategoryPrice)
                .collect(toList()));
        return result;
    }

    private CategoryPriceDTO mapCategoryPrice(CategoryPrice categoryPrice) {
        CategoryPriceDTO result = new CategoryPriceDTO();
        result.setFare(categoryPrice.getFare());
        result.setTaxes(categoryPrice.getTaxes());
        result.setTotal(categoryPrice.getTotal());
        result.setPassengerCategory(categoryPrice.getPassengerCategory());
        result.setQuantity(categoryPrice.getQuantity());
        result.setFareInfo(
                categoryPrice.getFareInfo().stream().map(this::mapFareInfo).collect(toList())
        );
        return result;
    }

    private FareInfoDTO mapFareInfo(FareInfo fareInfo) {
        FareInfoDTO result = new FareInfoDTO();
        result.setFareBasis(fareInfo.getFareBasis());
        result.setFareFamilyCode(fareInfo.getFareFamilyCode());
        result.setFlightId(fareInfo.getFlightId());
        return result;
    }

    private List<SegmentDTO> mapLegs(List<Segment> segments,
                                     Map<String, SegmentFare> fareTerms,
                                     CodeContext codeContext) {
        return segments.stream()
                .map((sl) -> createSegmentDTO(sl, fareTerms, codeContext))
                .collect(toList());
    }

    private SegmentDTO createSegmentDTO(Segment sl, Map<String, SegmentFare> fareTerms, CodeContext codeContext) {
        SegmentDTO result = new SegmentDTO();
        List<FlightDTO> flights = mapFlights(sl.getFlights(), codeContext);
        for (FlightDTO flight : flights) {
            SegmentFare terms = fareTerms.get(flight.getId());
            if (terms != null) {
                flight.setFareTerms(FareTermsHelper.fillFakeMilesProperties(terms));
            }
        }
        result.setFlights(flights);
        return result;
    }

    private List<FlightDTO> mapFlights(List<Flight> flights, CodeContext codeContext) {
        return flights.stream().map((fl) -> mapFlight(fl, codeContext)).collect(toList());
    }

    private FlightDTO mapFlight(
            Flight flight,
            CodeContext codeContext
    ) {
        TAirport departureAirport;
        TAirport arrivalAirport;
        TAirline marketingAirline;
        TAirline operatingAirline;
        List<FlightStopDTO> flightStops = createFlightStops(flight.getFlightStops(), codeContext);
        if (codeContext == CodeContext.IATA) {
            departureAirport = airportDictionary.getByIataCode(flight.getDepCode());
            arrivalAirport = airportDictionary.getByIataCode(flight.getArrCode());
            marketingAirline = airlineDictionary.getByIataCode(flight.getMarketingAirlineCode());
            operatingAirline = airlineDictionary.getByIataCode(flight.getOperatingAirlineCode());
        } else {
            throw new IllegalArgumentException(String.format("Unknown code context %s", codeContext));
        }
        return createFlightDTO(flight,
                departureAirport, arrivalAirport,
                marketingAirline, operatingAirline,
                flightStops, flight.getSeatsLeft()
        );
    }

    private List<FlightStopDTO> createFlightStops(List<FlightStop> stops, CodeContext codeContext) {
        List<FlightStopDTO> result = null;
        if (stops != null) {
            result = stops.stream().map((fs) -> {
                FlightStopDTO flightStopDTO = new FlightStopDTO();
                if (codeContext == CodeContext.IATA) {
                    flightStopDTO.setAirportId(airportDictionary.getByIataCode(fs.getAirportCode()).getId());
                } else {
                    throw new IllegalArgumentException(String.format("Unknown code context %s", codeContext));
                }
                flightStopDTO.setTerminal(fs.getTerminal());
                return flightStopDTO;
            }).collect(toList());
        }
        return result;
    }

    private FlightDTO createFlightDTO(Flight flight, TAirport departureAirport, TAirport arrivalAirport,
                                      TAirline marketingAirline,
                                      TAirline operatingAirline, List<FlightStopDTO> flightStops, Integer seatsLeft) {
        FlightDTO result = new FlightDTO();
        result.setFrom(departureAirport.getId());
        result.setTo(arrivalAirport.getId());
        result.setDeparture(flight.getDepartureDateTime());
        result.setArrival(flight.getArrivalDateTime());
        result.setMarketingAviaCompany(marketingAirline.getId());
        result.setOperatingAviaCompany(operatingAirline.getId());
        result.setNumber(flight.getFlightNumber());
        result.setId(flight.getId());
        result.setSeatsLeft(seatsLeft);
        result.setAircraftSeats(flight.getAircraftSeats());
        result.setStops(flightStops);
        return result;
    }

    public AviaPromoCampaignsDTO mapPromoCampaigns(AviaPromoCampaignsInfo promoCampaigns) {
        if (promoCampaigns == null) {
            return null;
        }
        AviaPromoCampaignsDTO promoCampaignsDTO = new AviaPromoCampaignsDTO();
        if (promoCampaigns.getPromo2020() != null) {
            promoCampaignsDTO.setPromo2020(AviaPromo2020DTO.builder()
                    .eligible(promoCampaigns.getPromo2020().getEligible())
                    .build());
        }
        promoCampaignsDTO.setPlusPromo2021(convertAeroflotPlusPromo2021(promoCampaigns.getPlusPromo2021()));
        return promoCampaignsDTO;
    }

    private AeroflotPlusPromo2021DTO convertAeroflotPlusPromo2021(AeroflotPlusPromoInfo plusPromo2021) {
        if (plusPromo2021 == null) {
            return null;
        }
        return AeroflotPlusPromo2021DTO.builder()
                .enabled(plusPromo2021.isEnabled())
                .totalPlusPoints(plusPromo2021.getPlusCodes().stream()
                        .mapToInt(AeroflotPlusPromoInfo.PlusCode::getPoints)
                        .sum())
                .build();
    }
}
