package ru.yandex.travel.api.services.avia.variants;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.model.search.PriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.avia.booking.service.dto.VariantCheckToken;
import ru.yandex.avia.booking.service.dto.VariantDTO;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.services.avia.legacy.AviaLegacyJsonMapper;
import ru.yandex.travel.api.services.avia.references.AviaReferenceJsonFactory;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
public class AviaVariantInfoJsonFactory {
    private final ObjectMapper objectMapper = AviaLegacyJsonMapper.objectMapper;
    private final AviaVariantDTOFactory variantDTOFactory;
    private final AviaReferenceJsonFactory referenceJsonFactory;

    public ObjectNode createPriceInfoNode(Money preliminaryPrice, Money firstCheckPrice) {
        ObjectNode priceInfo = objectMapper.createObjectNode().putObject("price_info");
        priceInfo.set("preliminary_price", objectMapper.valueToTree(preliminaryPrice));
        priceInfo.set("first_check_price", objectMapper.valueToTree(firstCheckPrice));
        return priceInfo;
    }

    public ObjectNode createVariantInfoNode(UUID checkId, Variant variant) {
        VariantCheckToken token = new VariantCheckToken(checkId, variant.getPriceInfo().getId());
        ObjectNode variantInfo = objectMapper.createObjectNode();
        VariantDTO variantDTO = variantDTOFactory.mapVariantDTO(token, variant);
        variantInfo.set("variant", objectMapper.valueToTree(variantDTO));
        variantInfo.set("reference", objectMapper.valueToTree(referenceJsonFactory.createReferenceNode(variantDTO)));
        return variantInfo;
    }

    public VariantDTO parseVariant(JsonNode variantInfo) {
        try {
            return objectMapper.treeToValue(variantInfo.get("variant"), VariantDTO.class);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public List<VariantDTO> parseAllVariants(JsonNode allVariants) {
        try {
            List<VariantDTO> variants = new ArrayList<>();
            for (JsonNode variant : allVariants) {
                variants.add(objectMapper.treeToValue(variant, VariantDTO.class));
            }
            return variants;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public JsonNode createAllVariantsNode(UUID checkId, Variant variant) {
        ArrayNode list = objectMapper.createArrayNode();
        if (variant.getAllTariffs() == null) {
            return list;
        }
        for (PriceInfo tariff : variant.getAllTariffs()) {
            Variant variantCopy = variant.toBuilder().priceInfo(tariff).allTariffs(null).build();
            VariantCheckToken token = new VariantCheckToken(checkId, variantCopy.getPriceInfo().getId());
            VariantDTO alternativeVariantDto = variantDTOFactory.mapVariantDTO(token, variantCopy);
            list.add(objectMapper.valueToTree(alternativeVariantDto));
        }
        return list;
    }
}
