package ru.yandex.travel.api.services.avia.variants;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;
import javax.money.CurrencyUnit;
import javax.money.Monetary;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionSystemException;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.avia.booking.partners.gateways.BookingGateway;
import ru.yandex.avia.booking.partners.gateways.BookingTestingScenarios;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData.SearchParams;
import ru.yandex.avia.booking.partners.gateways.model.availability.AvailabilityCheckRequest;
import ru.yandex.avia.booking.partners.gateways.model.availability.AvailabilityCheckResponse;
import ru.yandex.avia.booking.partners.gateways.model.availability.DepartureIsTooCloseException;
import ru.yandex.avia.booking.partners.gateways.model.availability.VariantNotAvailableException;
import ru.yandex.avia.booking.partners.gateways.model.search.PriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.avia.booking.promo.AeroflotPlusPromoInfo;
import ru.yandex.avia.booking.promo.AviaPromo2020Info;
import ru.yandex.avia.booking.promo.AviaPromoCampaignsInfo;
import ru.yandex.avia.booking.service.dto.VariantCheckResponseDTO;
import ru.yandex.avia.booking.service.dto.VariantCheckToken;
import ru.yandex.avia.booking.service.dto.VariantDTO;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonApiClient;
import ru.yandex.avia.booking.services.tdapi.AviaTicketDaemonSegment;
import ru.yandex.avia.booking.services.tdapi.InvalidationReason;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.exceptions.UnsupportedCurrencyException;
import ru.yandex.travel.api.services.avia.AviaBookingMeters;
import ru.yandex.travel.api.services.avia.AviaBookingProviderResolver;
import ru.yandex.travel.api.services.avia.fares.AviaFareFamilyService;
import ru.yandex.travel.api.services.avia.fares.AviaFareRulesException;
import ru.yandex.travel.api.services.avia.td.AviaTdInfo;
import ru.yandex.travel.api.services.avia.td.AviaTdInfoExtractor;
import ru.yandex.travel.api.services.avia.td.AviaTdPromoCampaigns;
import ru.yandex.travel.api.services.avia.td.AviaTdSegment;
import ru.yandex.travel.api.services.avia.td.promo.AviaTdAeroflotPlus2021Offer;
import ru.yandex.travel.api.services.avia.variants.model.AviaAvailabilityCheckState;
import ru.yandex.travel.api.services.avia.variants.model.AviaCachedVariantCheck;
import ru.yandex.travel.api.services.avia.variants.model.AviaVariantAvailabilityCheck;
import ru.yandex.travel.api.services.avia.variants.repositories.AviaVariantRepository;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirlineDictionary;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.commons.proto.TAviaTestContext;

import static java.util.stream.Collectors.toList;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
@Slf4j
public class AviaVariantService implements AutoCloseable {
    private static final Set<CurrencyUnit> SUPPORTED_CURRENCIES = Set.of(
            Monetary.getCurrency("RUB")
    );

    private final AviaVariantRepository variantRepository;
    private final AviaTdInfoExtractor ticketDaemonInfoExtractor;
    private final AviaBookingProviderResolver bookingGatewayResolver;
    private final AviaVariantInfoJsonFactory variantInfoJsonFactory;
    private final PlatformTransactionManager platformTransactionManager;
    private final AviaVariantCacheService variantsCacheService;
    private final AviaFareFamilyService fareFamilyService;
    private final AviaBookingMeters meters;
    private final AviaTicketDaemonApiClient tdApiClient;
    private final AviaAirlineDictionary airlineDictionary;
    private final AviaBookingProperties properties;
    private final Environment environment;

    private TransactionTemplate dbCacheLookupTxTemplate;
    private ExecutorService variantsLookupExecutor;

    @PostConstruct
    void init() {
        // we rely on the fact that we use PG which allows us to avoid phantom reads
        // under the ISOLATION_REPEATABLE_READ isolation level
        this.dbCacheLookupTxTemplate = new TransactionTemplate(platformTransactionManager);
        this.dbCacheLookupTxTemplate.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);

        // many long FlightPriceRQ requests (with retries) for the same destination can block in parallel,
        // we're moving them out into a separate thread poll to limit possible locks for other API calls
        variantsLookupExecutor = Executors.newFixedThreadPool(properties.getVariantCache().getMaxConcurrency(),
                new ThreadFactoryBuilder().setNameFormat("CheckAvailability-%d").build());
    }

    @Override
    public void close() {
        Duration cacheShutdownTimeout = properties.getVariantCache().getShutdownTimeout();
        MoreExecutors.shutdownAndAwaitTermination(variantsLookupExecutor,
                cacheShutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    public CompletableFuture<VariantCheckResponseDTO> checkAvailability(JsonNode ticketDaemonInfo) {
        CompletableFuture<VariantCheckResponseDTO> result = new CompletableFuture<>();
        result.completeAsync(() -> checkAvailability(ticketDaemonInfo, null), variantsLookupExecutor);
        return result;
    }

    public CompletableFuture<JsonNode> reCheckAvailability(UUID checkId) {
        try {
            log.info("re-checking availability: checkId={}", checkId);
            meters.getAvailabilityCheckRerun().increment();
            checkAvailability(getVariantInfo(checkId), checkId);
            return getVariantInfoFuture(checkId);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private VariantCheckResponseDTO checkAvailability(JsonNode ticketDaemonInfo, UUID checkIdToRefresh) {
        if (!ticketDaemonInfo.isObject()) {
            throw new IllegalArgumentException(
                    "Expected ticketDaemonInfo json object to be passed"
            );
        }
        AviaTdInfo tDaemonInfo = ticketDaemonInfoExtractor.parseTdaemonInfo(ticketDaemonInfo);
        BookingGateway bookingGateway = bookingGatewayResolver.gatewayForPartner(
                tDaemonInfo.getPartnerCode(), tDaemonInfo.getTestContext());
        Object variantInfo = bookingGateway.resolveVariantInfoAndOptimizeJson(ticketDaemonInfo);
        String variantId = bookingGateway.getExternalVariantId(variantInfo);

        int maxAttempts = properties.getVariantCache().getLockAttempts();
        for (int i = 0; i < maxAttempts; i++) {
            try {
                AviaVariantAvailabilityCheck check = dbCacheLookupTxTemplate.execute(
                        txStatus -> checkAvailabilityImpl(bookingGateway, tDaemonInfo, variantInfo,
                                checkIdToRefresh != null ? variantRepository.getOne(checkIdToRefresh) : null)
                );
                Preconditions.checkNotNull(check, "Availability check result should never be null");
                if (check.getState() == AviaAvailabilityCheckState.SUCCESS) {
                    meters.getVariantsAvailable().increment();
                    VariantCheckToken token = new VariantCheckToken(check.getId(), variantId);
                    return new VariantCheckResponseDTO(String.format("%s?token=%s",
                            properties.getBookingPageUrl(), token), token);
                } else if (check.getState() == AviaAvailabilityCheckState.EXTERNAL_REDIRECT) {
                    throw new AviaVariantNotSupportedException("External redirect is expected");
                } else {
                    meters.getNoVariantsAvailable().increment();
                    throw new AviaVariantsNotFoundException("Not available", check.getId());
                }
            } catch (AviaVariantCacheRecordLockException e) {
                log.debug("some parallel availability check is already in progress for this variant, " +
                        "waiting for its completion: variant_id={}", variantId);
                try {
                    // until there are more attempts left
                    if (i + 1 < maxAttempts) {
                        // randomizing delays to be in the [delay * 0,5, delay * 1.5) interval
                        // with the average value of delay * 1.0
                        long waitDelay = properties.getVariantCache().getLockRetryTimeout().toMillis();
                        long rndPart = (long) (Math.random() * waitDelay);
                        Thread.sleep(waitDelay / 2 + rndPart);
                    }
                } catch (InterruptedException ie) {
                    meters.getAvailabilityCheckOtherError().increment();
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("can't wait for the cached item lock anymore", ie);
                }
            } catch (Exception e) {
                if (!(e instanceof AviaVariantsNotFoundException)) {
                    meters.getAvailabilityCheckOtherError().increment();
                    log.warn("Unexpected check availability error", e);
                }
                throw e;
            }
        }
        meters.getAvailabilityCheckTimeout().increment();
        throw new RuntimeException("Can't start a new availability check or get cached results: variant_id=" + variantId);
    }

    private void ensureSupportedVariant(Object variant) {
        if (!(variant instanceof AeroflotVariant)) {
            return;
        }
        // todo(tlg-13): TRAVELBACK-234: use dynamic avia dictionaries instead
        // when done, substitute RuntimeExceptions with NoSuchElementExc. (5xx -> 4xx code) @ all avia dictionaries
        AeroflotVariant aeroflotVariant = (AeroflotVariant) variant;
        Preconditions.checkArgument(!aeroflotVariant.getSegments().isEmpty(),
                "Empty variant segments for " + aeroflotVariant.getOffer().getId());
    }

    private AviaVariantAvailabilityCheck checkAvailabilityImpl(BookingGateway bookingGateway, AviaTdInfo tDaemonInfo,
                                                               Object variantInfo,
                                                               AviaVariantAvailabilityCheck checkToRefresh) {
        ensureSupportedVariant(variantInfo);
        boolean isNew = checkToRefresh == null;
        AviaVariantAvailabilityCheck checkData = isNew ?
                AviaVariantAvailabilityCheck.createForData(tDaemonInfo.getRawData()) :
                checkToRefresh;
        String variantId = bookingGateway.getExternalVariantId(variantInfo);
        AviaCachedVariantCheck cached;
        try {
            cached = variantsCacheService.findOrCreateCachedCheck(tDaemonInfo.getPartnerCode(), variantId);
        } catch (TransactionSystemException e) {
            Throwable applicationException = e.getApplicationException();
            if (applicationException instanceof AviaVariantCacheRecordLockException) {
                throw (AviaVariantCacheRecordLockException) applicationException;
            }

            throw e;
        }
        if (cached.getCheckId() != null) {
            AviaVariantAvailabilityCheck check = variantRepository.getOne(cached.getCheckId());
            log.info("A cached availability check result has been found: id={}/{}, check.state={}",
                    cached.getPartnerId(), cached.getVariantId(), check.getState());
            if (checkToRefresh == null) {
                return check;
            } else {
                log.info("We're refreshing the existing check (variant_id={}), skipping the cached record",
                        checkToRefresh.getId());
            }
        }
        try {
            log.info("Checking availability for partner {}: variant_id={}",
                    cached.getPartnerId(), cached.getVariantId());
            AvailabilityCheckRequest request = AvailabilityCheckRequest.builder()
                    .token(variantInfo)
                    .build();

            AvailabilityCheckResponse response = bookingGateway.checkAvailabilityAll(request);

            addContextInfo(response.getVariant(), tDaemonInfo.getRawData());
            addPromoCampaigns(response.getVariant(), tDaemonInfo);

            // we will fail on unknown tariffs for the main offer but will just ignore (remove) alternative offers
            // with unknown tariffs
            fareFamilyService.checkUnknownFareCodesForRequestedVariant(response.getVariant());
            fareFamilyService.removeOffersWithUnknownFareCodes(response.getVariant());
            if (!variantId.equals(response.getVariant().getPriceInfo().getId())) {
                meters.getTariffChanged().increment();
            }
            if (!tDaemonInfo.getPreliminaryPrice().equals(response.getVariant().getPriceInfo().getTotal())) {
                invalidateTdVariantSafe(tDaemonInfo, InvalidationReason.PRICE_CHANGED);
            }
            // the same logic should apply to unsupported currencies
            ensureSupportedCurrency(response.getVariant());
            ensureSupportedCountry(response.getVariant());

            checkData.setState(AviaAvailabilityCheckState.SUCCESS);
            ObjectNode updatedData = addVariantInfo(checkData.getId(), tDaemonInfo.getRawData(), tDaemonInfo, response);
            bookingGateway.synchronizeUpdatedVariantInfoJson(updatedData, response.getVariant());
            if (properties.getEnableTestingScenarios() == Boolean.TRUE) {
                handleTestingScenarios(updatedData);
            }
            checkData.setData(updatedData);
            save(checkData, isNew);
            cached.setCheckId(checkData.getId());
            variantsCacheService.updateWithNewExpiration(cached);
            log.info("Variant is available: variantId={}", variantId);
            return checkData;
        } catch (VariantNotAvailableException e) {
            if (e instanceof DepartureIsTooCloseException) {
                // we can't book this variant but the user may still have a chance to do it via Aeroflot's site
                checkData.setState(AviaAvailabilityCheckState.EXTERNAL_REDIRECT);
            } else {
                checkData.setState(AviaAvailabilityCheckState.ERROR);
                invalidateTdVariantSafe(tDaemonInfo, InvalidationReason.VARIANT_NOT_AVAILABLE);
            }
            log.info("Variant is not available: variantId={}, e.msg={}", variantId, e.getMessage());
            save(checkData, isNew);
            cached.setCheckId(checkData.getId());
            variantsCacheService.updateWithNewExpiration(cached);
            return checkData;
        } catch (AviaFareRulesException e) {
            log.info("The variant can't be booked: variantId={}, e.msg={}, tdData={}", variantId, e.getMessage(),
                    tDaemonInfo.getRawData());
            throw e;
        } catch (Exception e) {
            log.info("Error checking availability; tdData={}", tDaemonInfo.getRawData(), e);
            throw e;
        }
    }

    private void save(AviaVariantAvailabilityCheck check, boolean isNew) {
        if (isNew) {
            check.setCreatedAt(LocalDateTime.now());
            check.setUpdatedAt(LocalDateTime.now());
            variantRepository.insert(check);
        } else {
            check.setUpdatedAt(LocalDateTime.now());
            variantRepository.update(check);
        }
    }

    private void ensureSupportedCurrency(Variant variant) {
        CurrencyUnit currency = variant.getPriceInfo().getTotal().getCurrency();
        if (!SUPPORTED_CURRENCIES.contains(currency)) {
            throw new UnsupportedCurrencyException("Unsupported currency: " + currency.getCurrencyCode() +
                    "; expected one of " + SUPPORTED_CURRENCIES.stream().map(CurrencyUnit::getCurrencyCode).collect(toList()));
        }
        List<PriceInfo> unsupported = new ArrayList<>();
        for (PriceInfo offer : variant.getAllTariffs()) {
            if (!SUPPORTED_CURRENCIES.contains(offer.getTotal().getCurrency())) {
                unsupported.add(offer);
            }
        }
        if (!unsupported.isEmpty()) {
            log.warn("Removing alternative offers with unsupported currencies: {}",
                    unsupported.stream().map(PriceInfo::getId).collect(toList()));
            variant.getAllTariffs().removeAll(unsupported);
        }
    }

    private void ensureSupportedCountry(Variant variant) {
        if (!"RU".equals(variant.getCountryOfSale())) {
            throw new IllegalArgumentException("Only RU is a supported country of sale at the moment; received=" + variant.getCountryOfSale());
        }
    }

    private void addContextInfo(Variant variant, JsonNode ticketDaemonInfo) {
        ObjectNode updatedData = (ObjectNode) ticketDaemonInfo;
        // todo(tlg-13) need to find a better way to pass the lang context, or not to do it at all
        String country = updatedData.at("/redirect_data/order_data/booking_info/CountryCode").textValue();
        String lang = updatedData.at("/redirect_data/order_data/booking_info/LanguageCode").textValue();
        variant.setCountryOfSale(!Strings.isNullOrEmpty(country) ? country : "RU");
        variant.setLang(!Strings.isNullOrEmpty(lang) ? lang : "ru");
    }

    private ObjectNode addVariantInfo(UUID checkId, JsonNode ticketDaemonInfo, AviaTdInfo tdInfo,
                                      AvailabilityCheckResponse response) {
        Preconditions.checkNotNull(checkId, "checkId can't be null");
        Money preliminaryPrice = tdInfo.getPreliminaryPrice();
        Money actualPrice = response.getVariant().getPriceInfo().getTotal();
        if (!preliminaryPrice.equals(actualPrice)) {
            meters.getPriceChanged().increment();
        }
        ObjectNode updatedData = (ObjectNode) ticketDaemonInfo;
        updatedData.set("price_info",
                variantInfoJsonFactory.createPriceInfoNode(
                        preliminaryPrice,
                        actualPrice
                )
        );
        updatedData.set("variant_info", variantInfoJsonFactory.createVariantInfoNode(
                checkId, response.getVariant()));
        if (response.getVariant().hasMultipleTariffs()) {
            updatedData.set("all_variants", variantInfoJsonFactory.createAllVariantsNode(
                    checkId, response.getVariant()));
        }
        return updatedData;
    }

    public VariantDTO parseVariant(JsonNode checkData) {
        return variantInfoJsonFactory.parseVariant(checkData.get("variant_info"));
    }

    public List<VariantDTO> parseAllVariants(JsonNode checkData) {
        return variantInfoJsonFactory.parseAllVariants(checkData.get("all_variants"));
    }

    private void addPromoCampaigns(Variant variant, AviaTdInfo tdInfo) {
        variant.getPriceInfo().setPromoCampaigns(getPromoCampaignsInfo(tdInfo, variant.getPriceInfo().getId()));
        for (PriceInfo tariff : variant.getAllTariffs()) {
            tariff.setPromoCampaigns(getPromoCampaignsInfo(tdInfo, tariff.getId()));
        }
    }

    public AviaPromoCampaignsInfo getPromoCampaignsInfo(AviaTdInfo tdInfo, String offerId) {
        return AviaPromoCampaignsInfo.builder()
                .promo2020(convertAeroflot2020Promo(tdInfo.getPromoCampaigns().getPromo2020Ids(), offerId))
                .plusPromo2021(convertAeroflotPlusPromo(tdInfo.getPromoCampaigns(), offerId))
                .build();
    }

    private AviaPromo2020Info convertAeroflot2020Promo(Collection<String> promo2020Ids, String offerId) {
        if (promo2020Ids == null) {
            return null;
        }
        return AviaPromo2020Info.builder()
                .eligible(promo2020Ids.contains(offerId))
                .build();
    }

    private AeroflotPlusPromoInfo convertAeroflotPlusPromo(AviaTdPromoCampaigns promoCampaigns, String offerId) {
        List<AviaTdAeroflotPlus2021Offer> promoOffers = promoCampaigns.getAeroflotPlusPromo2021Offers();
        if (promoOffers == null) {
            return null;
        }
        return promoOffers.stream()
                .filter(o -> offerId.equals(o.getOfferId()))
                .map(o -> AeroflotPlusPromoInfo.builder()
                        .enabled(true)
                        .plusCodes(o.getPlusCodes().stream()
                                .map(c -> AeroflotPlusPromoInfo.PlusCode.builder()
                                        .points(c.getPoints())
                                        .build())
                                .collect(toList()))
                        .build())
                .collect(CustomCollectors.atMostOne());
    }

    public JsonNode getVariantInfo(UUID variantId) {
        return dbCacheLookupTxTemplate.execute(status -> variantRepository.getOne(variantId).getData());
    }

    public CompletableFuture<JsonNode> getVariantInfoFuture(UUID variantId) {
        try {
            return CompletableFuture.completedFuture(getVariantInfo(variantId));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private void handleTestingScenarios(ObjectNode variantData) {
        String testScenario = variantData.path("additional_data").path("utm_source").asText();
        if (BookingTestingScenarios.AVAILABILITY_CHECK_NOT_FOUND.equals(testScenario)) {
            throw new VariantNotAvailableException("testing scenario: NOT FOUND");
        }
        if (BookingTestingScenarios.AVAILABILITY_CHECK_NEW_PRICE.equals(testScenario)) {
            ObjectNode preliminaryPrice = (ObjectNode) variantData.get("price_info").get("preliminary_price");
            ObjectNode checkPrice = (ObjectNode) variantData.get("price_info").get("first_check_price");
            preliminaryPrice.put("value", (int) (checkPrice.get("value").asInt() * 0.95));
        }
        if (BookingTestingScenarios.AVAILABILITY_CHECK_UNHANDLED_ERROR.equals(testScenario)) {
            throw new RuntimeException("testing scenario: UNHANDLED ERROR");
        }
    }

    private void invalidateTdVariantSafe(AviaTdInfo tdInfo, InvalidationReason reason) {
        try {
            TAviaTestContext testContext = tdInfo.getTestContext();
            if (testContext != null) {
                log.debug("Ignoring TD variant invalidation request as a TestContext is present, variant qid {}",
                        tdInfo.getVariantQid());
                return;
            }
            invalidateTdVariant(tdInfo, reason);
        } catch (Exception e) {
            log.warn("failed to invalidate td variant: {}", tdInfo.getVariantQid(), e);
        }
    }

    private void invalidateTdVariant(AviaTdInfo tdInfo, InvalidationReason reason) {
        SearchParams sp = tdInfo.getSearchParams();
        List<List<AviaTicketDaemonSegment>> legs = new ArrayList<>();
        for (AviaTdSegment leg : tdInfo.getSegments()) {
            legs.add(leg.getSegments().stream()
                    .map(seg -> new AviaTicketDaemonSegment(
                            airlineDictionary.getById(seg.getMarketingAirlineCode()).getIataCode(),
                            seg.getFlightNumber(),
                            seg.getDepartureDateTime()
                    ))
                    .collect(toList()));
        }
        Preconditions.checkArgument(legs.size() >= 1 && legs.size() <= 2,
                "Exactly 1 or 2 legs expected; legs=%s", legs);
        tdApiClient.invalidateVariant(
                AviaTicketDaemonApiClient.InvalidateVariantParams.builder()
                        .nationalVersion(sp.getNationalVersion())
                        .lang(sp.getLang())
                        .klass(sp.getKlass())
                        .pointFrom(sp.getPointFrom())
                        .pointTo(sp.getPointTo())
                        .adults(tdInfo.getAdultCount())
                        .children(tdInfo.getChildrenCount())
                        .infants(tdInfo.getInfantCount())
                        .partner(tdInfo.getPartnerCode())
                        .forward(legs.get(0))
                        .backward(legs.size() > 1 ? legs.get(1) : null)
                        .variantTag(tdInfo.getVariantTag())
                        .build(),
                reason
        );
    }
}
