package ru.yandex.travel.api.services.avia.variants.repositories;

import java.io.IOException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.postgresql.util.PGobject;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.services.avia.variants.model.AviaAvailabilityCheckState;
import ru.yandex.travel.api.services.avia.variants.model.AviaVariantAvailabilityCheck;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@Transactional(propagation = Propagation.MANDATORY)
@RequiredArgsConstructor
@Slf4j
public class AviaVariantRepository {
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final JdbcTemplate jdbcTemplate;

    public AviaVariantAvailabilityCheck getOne(UUID id) {
        List<AviaVariantAvailabilityCheck> result = jdbcTemplate.query(
                "select data, created_at, updated_at, state from variant_availability_checks where id = ?",
                new Object[]{id},
                (rs, i) -> {
                    JsonNode data = convertJsonFromDb(rs.getObject("data"));
                    Timestamp createdAt = rs.getTimestamp("created_at");
                    Timestamp updatedAt = rs.getTimestamp("updated_at");
                    String state = rs.getString("state");
                    return AviaVariantAvailabilityCheck.builder()
                            .id(id)
                            .data(data)
                            .createdAt(createdAt != null ? createdAt.toLocalDateTime() : null)
                            .updatedAt(updatedAt != null ? updatedAt.toLocalDateTime() : null)
                            .state(!Strings.isNullOrEmpty(state) ? AviaAvailabilityCheckState.fromString(state) : null)
                            .build();
                });
        if (result.size() == 0) {
            throw new NoSuchElementException("Variant availability check not found: id=" + id);
        }
        Preconditions.checkState(result.size() == 1, "Too many results for a getById request: results=%s", result);
        return result.get(0);
    }

    public void insert(AviaVariantAvailabilityCheck check) {
        String jsonData = serializeJsonNode(check.getData());
        int updated = jdbcTemplate.update(
                "insert into variant_availability_checks(id, data, created_at, updated_at, state) values(?, ?, ?, ?, ?)",
                ps -> {
                    AviaAvailabilityCheckState state = check.getState();
                    ps.setObject(1, check.getId());
                    ps.setObject(2, jsonData, Types.OTHER);
                    ps.setTimestamp(3, Timestamp.valueOf(check.getCreatedAt()));
                    ps.setTimestamp(4, Timestamp.valueOf(check.getUpdatedAt()));
                    ps.setString(5, state != null ? state.getValue() : null);
                });
        Preconditions.checkState(updated == 1, "Exactly 1 inserted row is expected but got %s", updated);
    }

    public void update(AviaVariantAvailabilityCheck check) {
        String jsonData = serializeJsonNode(check.getData());
        int updated = jdbcTemplate.update("update variant_availability_checks " +
                        "set data = ?, created_at = ?, updated_at = ?, state = ? " +
                        "where id = ?",
                ps -> {
                    AviaAvailabilityCheckState state = check.getState();
                    ps.setObject(1, jsonData, Types.OTHER);
                    ps.setTimestamp(2, Timestamp.valueOf(check.getCreatedAt()));
                    ps.setTimestamp(3, Timestamp.valueOf(check.getUpdatedAt()));
                    ps.setString(4, state != null ? state.getValue() : null);
                    ps.setObject(5, check.getId());
                });
        Preconditions.checkState(updated == 1, "Exactly 1 updated row is expected but got %s", updated);
    }

    private String serializeJsonNode(JsonNode value) {
        try {
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private JsonNode deserializeJsonNode(String value) {
        try {
            return objectMapper.readTree(value);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private JsonNode convertJsonFromDb(Object value) {
        if (value == null) {
            return null;
        }
        if (value instanceof String) {
            return deserializeJsonNode((String) value);
        }
        if (value instanceof PGobject) {
            return deserializeJsonNode(((PGobject) value).getValue());
        }
        throw new IllegalArgumentException("Unknown object type (excepted pgobject or json string); got " + value.getClass().getName());
    }
}
