package ru.yandex.travel.api.services.buses;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.CustomConverter;
import ma.glasnost.orika.CustomMapper;
import ma.glasnost.orika.MapperFacade;
import ma.glasnost.orika.MappingContext;
import ma.glasnost.orika.MappingContextFactory;
import ma.glasnost.orika.converter.ConverterFactory;
import ma.glasnost.orika.impl.DefaultMapperFactory;
import ma.glasnost.orika.metadata.Type;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.buses.req_rsp.BusDirectionRspV1;
import ru.yandex.travel.api.models.common.Price;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlockData;
import ru.yandex.travel.api.models.rasp.morda_backend.ParseContext;
import ru.yandex.travel.api.models.train.CrossLink;
import ru.yandex.travel.api.spec.buses.PointType;
import ru.yandex.travel.bus.model.BusPointsHttp;
import ru.yandex.travel.bus.model.BusRideHttp;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;

@Component
@Slf4j
public class BusDirectionMapper {
    interface Properties {
        String POINTS_MAP = "POINTS_MAP";
        String FROM_TIMEZONE = "FROM_TIMEZONE";
        String TO_TIMEZONE = "TO_TIMEZONE";
    }

    public static final Map<ProtoCurrencyUnit, Price.Currency> CURRENCY_MAP = ImmutableMap.of(
        ProtoCurrencyUnit.RUB, Price.Currency.RUB,
        ProtoCurrencyUnit.USD, Price.Currency.USD,
        ProtoCurrencyUnit.EUR, Price.Currency.EUR,
        ProtoCurrencyUnit.JPY, Price.Currency.JPY
    );

    public BusDirectionRspV1 map(
            ParseContext parseContext,
            String day,
            BusRideHttp[] busRides,
            BusPointsHttp busPoints,
            CrossLink[] crossLinks,
            CrosslinksHotelsBlockData hotelsBlock) {
        MappingContextFactory mappingContextFactory = new MappingContext.Factory();
        DefaultMapperFactory mapperFactory = new DefaultMapperFactory
                .Builder()
                .mappingContextFactory(mappingContextFactory)
                .build();

        Map<String, BusPointsHttp.Point> pointMap = Arrays.stream(busPoints.getData())
                .collect(Collectors.toMap(BusPointsHttp.Point::getCode, Function.identity()));

        MappingContext mappingContext = mappingContextFactory.getContext();
        mappingContext.setProperty(Properties.POINTS_MAP, pointMap);
        mappingContext.setProperty(Properties.FROM_TIMEZONE, parseContext.getFrom().getTimezone());
        mappingContext.setProperty(Properties.TO_TIMEZONE, parseContext.getTo().getTimezone());

        final ConverterFactory converterFactory = mapperFactory.getConverterFactory();

        converterFactory.registerConverter("money2price", new CustomConverter<Money, Price>() {
            @Override
            public Price convert(Money source,  Type<? extends Price> destinationType, MappingContext mappingContext) {
                return new Price(source.getNumberStripped().toPlainString(),
                        CURRENCY_MAP.get((ProtoCurrencyUnit) source.getCurrency()));
            }
        });

        converterFactory.registerConverter("error2error", new CustomConverter<ParseContext.ResultError[], String[]>() {
            @Override
            public String[] convert(ParseContext.ResultError[] source,  Type<? extends String[]> destinationType, MappingContext mappingContext) {
                return Arrays.stream(source).map(error -> error.getType()).collect(Collectors.toList()).toArray(new String[0]);
            }
        });

        mapperFactory.classMap(BusRideHttp.class, BusDirectionRspV1.BusesSegment.class)
                .exclude("price")
                .exclude("from")
                .exclude("to")
                .exclude("arrivalTime")
                .exclude("departureTime")
                .exclude("duration")
                .byDefault()
                .fieldMap("price", "price").aToB().converter("money2price").add()
                .customize(
                        new CustomMapper<BusRideHttp, BusDirectionRspV1.BusesSegment>() {
                            @Override
                            public void mapAtoB(BusRideHttp busRide, BusDirectionRspV1.BusesSegment segment,
                                                MappingContext context) {

                                var mapPoint = (Map<String, BusPointsHttp.Point>) context.getProperty(Properties.POINTS_MAP);
                                String fromTimezone = (String) context.getProperty(Properties.FROM_TIMEZONE);
                                String toTimezone = (String) context.getProperty(Properties.TO_TIMEZONE);

                                BusDirectionRspV1.BusesApiPoint busesApiFromPoint = mapRidePoint2ApiPoint(
                                        busRide.getFrom(),
                                        busRide.getFromDesc(),
                                        fromTimezone,
                                        mapPoint);

                                BusDirectionRspV1.BusesApiPoint busesApiToPoint = mapRidePoint2ApiPoint(
                                        busRide.getTo(),
                                        busRide.getToDesc(),
                                        toTimezone,
                                        mapPoint);

                                segment.setFrom(busesApiFromPoint);
                                segment.setTo(busesApiToPoint);

                                // departureTime and arrivalTime are unix timestamps in local timezone (not UTC)
                                Instant fromEpoch = Instant.ofEpochSecond(busRide.getDepartureTime());
                                LocalDateTime fromDateTime = LocalDateTime.ofInstant(fromEpoch, ZoneOffset.UTC);
                                segment.setDepartureTime(fromDateTime);

                                Optional.ofNullable(busRide.getArrivalTime())
                                        .map(Instant::ofEpochSecond)
                                        .map(toEpoch -> LocalDateTime.ofInstant(toEpoch, ZoneOffset.UTC))
                                        .ifPresent(segment::setArrivalTime);

                                segment.setDuration(busRide.getDuration() * 1000);
                            }
                        }
                )
                .register();

        mapperFactory.classMap(ParseContext.Point.class, BusDirectionRspV1.ContextPoint.class)
                .customize(new CustomMapper<>() {
                    @Override
                    public void mapAtoB(ParseContext.Point point, BusDirectionRspV1.ContextPoint geoPoint,  MappingContext context) {
                        var titleLinguistics = new BusDirectionRspV1.ContextPointLinguistics();
                        titleLinguistics.setAccusativeCase(point.getTitleAccusative());
                        titleLinguistics.setGenitiveCase(point.getTitleGenitive());
                        titleLinguistics.setNominativeCase(point.getTitle());
                        titleLinguistics.setLocativeCase(point.getTitleLocative());

                        var settlement = new BusDirectionRspV1.ContextPointSettlement();
                        ParseContext.Point.Settlement pointSettlement = point.getSettlement();
                        if (pointSettlement != null) {
                            settlement.setTitle(pointSettlement.getTitle());
                        }

                        var country = new BusDirectionRspV1.ContextPointCountry();
                        ParseContext.Point.Country pointCountry = point.getCountry();
                        if (pointCountry != null) {
                            country.setCode(pointCountry.getCode());
                            country.setTitle(pointCountry.getTitle());
                        }

                        var region = new BusDirectionRspV1.ContextPointRegion();
                        region.setTitle(point.getTitle());

                        geoPoint.setKey(point.getKey());
                        geoPoint.setSlug(point.getSlug());
                        geoPoint.setTimezone(point.getTimezone());
                        geoPoint.setSettlement(settlement);
                        geoPoint.setRegion(region);
                        geoPoint.setCountry(country);
                        geoPoint.setTitle(titleLinguistics);
                    }
                })
                .register();

        mapperFactory.classMap(ParseContext.class, BusDirectionRspV1.BusesContext.class)
                .exclude("errors")
                .byDefault()
                .fieldMap("errors", "errors").converter("error2error").add()
                .register();

        mapperFactory.classMap(CrossLink.class, BusDirectionRspV1.CrossLink.class)
                .field("toTitleRuNominative", "toTitle" )
                .field("fromTitleRuNominative", "fromTitle")
                .byDefault()
                .register();

        MapperFacade mapper = mapperFactory.getMapperFacade();
        BusDirectionRspV1.BusesContext context = mapper.map(parseContext, BusDirectionRspV1.BusesContext.class, mappingContext);
        List<BusDirectionRspV1.TDirectionPageBlock> blocks = new ArrayList<>();

        if (busRides != null && busRides.length > 0) {
            List<BusDirectionRspV1.BusesSegment> rides = mapper.mapAsList(busRides, BusDirectionRspV1.BusesSegment.class, mappingContext);
            var segmentsBlock = new BusDirectionRspV1.BusesSegmentsBlock(new BusDirectionRspV1.BusesSegmentsBlockData(null, day, rides));
            blocks.add(segmentsBlock);
        }

        if (crossLinks != null && crossLinks.length > 0) {
            List<BusDirectionRspV1.CrossLink> links = mapper.mapAsList(crossLinks, BusDirectionRspV1.CrossLink.class, mappingContext);
            var crossLinksBlock = new BusDirectionRspV1.BusesCrossLinksBlock(new BusDirectionRspV1.BusesCrossLinksBlockData(null, links));
            blocks.add(crossLinksBlock);
        }

        if (hotelsBlock != null) {
            var crossLinksHotelsBlock = new BusDirectionRspV1.CrossSaleHotelsBlock(hotelsBlock);
            blocks.add(crossLinksHotelsBlock);
        }

        return new BusDirectionRspV1(context, blocks);
    }

    private BusDirectionRspV1.BusesApiPoint mapRidePoint2ApiPoint(
            BusRideHttp.Point ridePoint,
            String defaultTitle,
            String defaultTimezone,
            Map<String, BusPointsHttp.Point> mapPoint
    ) {
        if (ridePoint == null) {
            ridePoint = new BusRideHttp.Point();
        }

        var fromType = ridePoint.getType();
        var fromId = ridePoint.getId();

        var busesApiPoint = new BusDirectionRspV1.BusesApiPoint();

        if (fromType == null || fromId == null) {
            busesApiPoint.setTitle(defaultTitle);
            busesApiPoint.setTimezone(defaultTimezone);
            return busesApiPoint;
        }

        String prefix = ridePoint.getType() == PointType.Enum.STATION.getNumber() ? "s" : "c";
        String code = prefix + String.valueOf(ridePoint.getId());
        BusPointsHttp.Point point = mapPoint.get(code);
        if (point == null) {
            busesApiPoint.setTitle(defaultTitle);
            busesApiPoint.setTimezone(defaultTimezone);
            return busesApiPoint;
        }

        String title = point.getName();
        if (title == null) {
            title = defaultTitle;
        }
        String timezone = point.getTimezone();
        if (timezone == null) {
            timezone = defaultTimezone;
        }

        busesApiPoint.setTitle(title);
        busesApiPoint.setTimezone(timezone);
        busesApiPoint.setFullTitle(point.getDescription());
        if (point.getLatitude() != null && point.getLongitude() != null) {
            var coords = new BusDirectionRspV1.Coordinates(point.getLatitude(), point.getLongitude());
            busesApiPoint.setCoordinates(coords);
        }

        return busesApiPoint;
    }
}
