package ru.yandex.travel.api.services.buses;

import java.time.Instant;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.dictionaries.country.CountryDataProvider;
import ru.yandex.travel.api.services.orders.BusModelMapService;
import ru.yandex.travel.api.spec.Currency;
import ru.yandex.travel.api.spec.DocumentType;
import ru.yandex.travel.api.spec.Gender;
import ru.yandex.travel.api.spec.LegalEntity;
import ru.yandex.travel.api.spec.MoneyAmount;
import ru.yandex.travel.api.spec.RegisterType;
import ru.yandex.travel.api.spec.buses.BenefitType;
import ru.yandex.travel.api.spec.buses.BookParams;
import ru.yandex.travel.api.spec.buses.CreateRideOfferRequest;
import ru.yandex.travel.api.spec.buses.CreateRideOfferResponse;
import ru.yandex.travel.api.spec.buses.Fare;
import ru.yandex.travel.api.spec.buses.Place;
import ru.yandex.travel.api.spec.buses.PlaceStatus;
import ru.yandex.travel.api.spec.buses.PlaceType;
import ru.yandex.travel.api.spec.buses.Ride;
import ru.yandex.travel.api.spec.buses.TCountry;
import ru.yandex.travel.api.spec.buses.TicketType;
import ru.yandex.travel.api.spec.buses.TicketTypeType;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.buses.backend.proto.EBenefitType;
import ru.yandex.travel.buses.backend.proto.EDocumentType;
import ru.yandex.travel.buses.backend.proto.EGenderType;
import ru.yandex.travel.buses.backend.proto.EPlaceStatus;
import ru.yandex.travel.buses.backend.proto.EPlaceType;
import ru.yandex.travel.buses.backend.proto.ERegisterType;
import ru.yandex.travel.buses.backend.proto.ETicketType;
import ru.yandex.travel.buses.backend.proto.TBookParams;
import ru.yandex.travel.buses.backend.proto.TCarrier;
import ru.yandex.travel.buses.backend.proto.TCitizenship;
import ru.yandex.travel.buses.backend.proto.TDocumentType;
import ru.yandex.travel.buses.backend.proto.TGender;
import ru.yandex.travel.buses.backend.proto.TPlace;
import ru.yandex.travel.buses.backend.proto.TPointKey;
import ru.yandex.travel.buses.backend.proto.TRide;
import ru.yandex.travel.buses.backend.proto.TSeat;
import ru.yandex.travel.buses.backend.proto.TSupplier;
import ru.yandex.travel.buses.backend.proto.TTicketType;
import ru.yandex.travel.buses.proto.TLabelParams;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;

import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.consumeNonEmpty;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.consumeNotNull;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.instantToProtoWithoutTimezone;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.moneyToProto;

@Service
@Slf4j
public class BusesApiService {

    public BusesApiService(CountryDataProvider countryDataProvider,
                           BusesService busesService,
                           @Qualifier("busesServiceExp") BusesService busesServiceExp,
                           BusModelMapService busModelMapService
    ) {
        this.countryDataProvider = countryDataProvider;
        this.service = busesService;
        this.serviceExp = busesServiceExp;
        this.busModelMapService = busModelMapService;
    }

    private final CountryDataProvider countryDataProvider;

    private static final Map<ERegisterType, RegisterType.Enum> REGISTER_TYPE_MAP =
            Map.of(
                    ERegisterType.REGISTER_TYPE_COMPANY, RegisterType.Enum.COMPANY,
                    ERegisterType.REGISTER_TYPE_ENTREPRENEUR, RegisterType.Enum.ENTREPRENEUR
            );
    private static final Map<EBenefitType, BenefitType.Enum> BENEFIT_TYPE_MAP =
            Map.ofEntries(
                    Map.entry(EBenefitType.BENEFIT_TYPE_COFFEE, BenefitType.Enum.COFFEE),
                    Map.entry(EBenefitType.BENEFIT_TYPE_CHARGER, BenefitType.Enum.CHARGER),
                    Map.entry(EBenefitType.BENEFIT_TYPE_PRESS, BenefitType.Enum.PRESS),
                    Map.entry(EBenefitType.BENEFIT_TYPE_TV, BenefitType.Enum.TV),
                    Map.entry(EBenefitType.BENEFIT_TYPE_WI_FI, BenefitType.Enum.WI_FI),
                    Map.entry(EBenefitType.BENEFIT_TYPE_NO_TICKET_REQUIRED, BenefitType.Enum.NO_TICKET_REQUIRED),
                    Map.entry(EBenefitType.BENEFIT_TYPE_WC, BenefitType.Enum.WC),
                    Map.entry(EBenefitType.BENEFIT_TYPE_CONDITIONER, BenefitType.Enum.CONDITIONER),
                    Map.entry(EBenefitType.BENEFIT_TYPE_COMMON_AUDIO, BenefitType.Enum.COMMON_AUDIO)
            );
    public static final Map<EDocumentType, DocumentType.Enum> DOCUMENT_TYPE_MAP =
            Map.ofEntries(
                    Map.entry(EDocumentType.DOCUMENT_TYPE_RU_PASSPORT, DocumentType.Enum.ru_national_passport),
                    Map.entry(EDocumentType.DOCUMENT_TYPE_RU_BIRTH_CERTIFICATE, DocumentType.Enum.ru_birth_certificate),
                    Map.entry(EDocumentType.DOCUMENT_TYPE_RU_INTERNATIONAL_PASSPORT,
                            DocumentType.Enum.ru_foreign_passport),
                    Map.entry(EDocumentType.DOCUMENT_TYPE_FOREIGN_PASSPORT, DocumentType.Enum.other)
            );
    private static final Map<ETicketType, TicketTypeType.Enum> TICKET_TYPE_MAP =
            Map.ofEntries(
                    Map.entry(ETicketType.TICKET_TYPE_FULL, TicketTypeType.Enum.FULL),
                    Map.entry(ETicketType.TICKET_TYPE_CHILD, TicketTypeType.Enum.CHILD),
                    Map.entry(ETicketType.TICKET_TYPE_BAGGAGE, TicketTypeType.Enum.BAGGAGE)
            );
    private static final Map<EGenderType, Gender.Enum> GENDER_MAP =
            Map.ofEntries(
                    Map.entry(EGenderType.GENDER_TYPE_MALE, Gender.Enum.male),
                    Map.entry(EGenderType.GENDER_TYPE_FEMALE, Gender.Enum.female)
            );
    private static final Map<EPlaceType, PlaceType.Enum> PLACE_TYPE_MAP =
            Map.ofEntries(
                    Map.entry(EPlaceType.PLACE_TYPE_DRIVER, PlaceType.Enum.DRIVER),
                    Map.entry(EPlaceType.PLACE_TYPE_SEAT, PlaceType.Enum.SEAT),
                    Map.entry(EPlaceType.PLACE_TYPE_PASSAGE, PlaceType.Enum.PASSAGE)
            );
    private static final Map<EPlaceStatus, PlaceStatus.Enum> PLACE_STATUS_MAP =
            Map.ofEntries(
                    Map.entry(EPlaceStatus.PLACE_STATUS_FREE, PlaceStatus.Enum.FREE),
                    Map.entry(EPlaceStatus.PLACE_STATUS_OCCUPIED, PlaceStatus.Enum.OCCUPIED)
            );
    private final BusesService service;
    private final BusesService serviceExp;
    private final BusModelMapService busModelMapService;

    public CompletableFuture<CreateRideOfferResponse> createRideOffer(@NotNull CreateRideOfferRequest request,
                                                                      @NotNull CommonHttpHeaders headers) {
        boolean useExp = headers.getExperiments().containsKey("BUSES_use_exp_backend");
        BusesService chosenService = useExp ? serviceExp : service;
        return chosenService.rideDetails(
                ru.yandex.travel.buses.backend.proto.api.CreateRideOfferRequest.newBuilder()
                        .setRideId(request.getRideId())
                        .setLabelParams(TLabelParams.newBuilder()
                                .setSerpReqId(request.getLabelParams().getSerpReqId())
                                .setUtmSource(request.getLabelParams().getUtmSource())
                                .setUtmMedium(request.getLabelParams().getUtmMedium())
                                .setUtmCampaign(request.getLabelParams().getUtmCampaign())
                                .setUtmTerm(request.getLabelParams().getUtmTerm())
                                .setUtmContent(request.getLabelParams().getUtmContent())
                                .setFrom(request.getLabelParams().getFrom())
                                .setGclid(request.getLabelParams().getGclid())
                                .setICookie(request.getLabelParams().getIcookie())
                                .setSerpUuid(request.getLabelParams().getSerpUuid())
                                .setTestBuckets(request.getLabelParams().getTestBuckets())
                                .setDevice(request.getLabelParams().getDevice())
                                .setTerminal(request.getLabelParams().getTerminal())
                                .setIp(request.getLabelParams().getIp())
                                .setRegionId(request.getLabelParams().getRegionId())
                                .setUid(request.getLabelParams().getUid())
                                .setYandexUid(request.getLabelParams().getYandexUid())
                                .setWizardReqId(request.getLabelParams().getWizardReqId())
                                .setSerpTestId(request.getLabelParams().getSerpTestId())
                                .setYtpReferer(request.getLabelParams().getYtpReferer())
                                .setYclid(request.getLabelParams().getYclid())
                                .setFbclid(request.getLabelParams().getFbclid())
                                .setMetrikaClientId(request.getLabelParams().getMetrikaClientId())
                                .setClid(request.getLabelParams().getClid())
                                .setAdmitadUid(request.getLabelParams().getAdmitadUid())
                                .setTravelpayoutsUid(request.getLabelParams().getTravelpayoutsUid())
                                .setVid(request.getLabelParams().getVid())
                                .build())
                        .build()
        ).thenApply(
                serviceRsp -> {
                    var builder = CreateRideOfferResponse.newBuilder();

                    consumeNonEmpty(builder::setError, serviceRsp.getError());

                    var offer = serviceRsp.getOffer();
                    if (ProtoUtils.hasFields(offer)) {
                        builder.setRide(convertRide(offer.getRide(), offer.getQueryFrom(), offer.getQueryTo()));
                        builder.setBookParams(convertBookParams(offer.getBookParams(), offer.getRide().getFee()));
                    }

                    consumeNonEmpty(builder::setOfferId, serviceRsp.getOfferId());
                    consumeNotNull(builder::setLabel, serviceRsp.getLabelHash());

                    return builder.build();
                }
        );
    }

    @NotNull
    private Ride convertRide(@NotNull TRide ride, TPointKey queryFrom, TPointKey queryTo) {
        var builder = Ride.newBuilder();

        builder.setId(ride.getId());

        builder.setDeparture(convertSeconds(ride.getDepartureTime()));
        var arrivalTime = ride.getArrivalTime();
        if (arrivalTime != 0) {
            builder.setArrival(convertSeconds(arrivalTime));
            builder.setDuration(ride.getDuration() * 1000);
        }

        builder.setPointFrom(busModelMapService.convertPoint(
                busModelMapService.makePointInfo(ride.getFromDesc(), ride.getFrom())));
        builder.setPointTo(busModelMapService.convertPoint(
                busModelMapService.makePointInfo(ride.getToDesc(), ride.getTo())));
        builder.setTitlePointFrom(busModelMapService.convertPoint(
                busModelMapService.makeTitlePointInfo(ride.getFromDesc(), ride.getFrom(), queryFrom)));
        builder.setTitlePointTo(busModelMapService.convertPoint(
                busModelMapService.makeTitlePointInfo(ride.getToDesc(), ride.getTo(), queryTo)));

        consumeNonEmpty(builder::setBusDescription, ride.getBus());
        consumeNonEmpty(builder::setRouteNumber, ride.getRouteNumber());
        consumeNonEmpty(builder::setRouteName, ride.getRouteName());
        builder.addAllBenefits(ride.getBenefitsList().stream().map(this::convertBenefit)::iterator);

        var supplier = ride.getSupplier();
        if (ProtoUtils.hasFields(supplier)) {
            builder.setSupplier(convertSupplier(supplier));
        }

        var carrier = ride.getCarrier();
        if (ProtoUtils.hasFields(carrier)) {
            builder.setCarrier(convertCarrier(carrier));
        } else { // TODO BUSES-1579 remove else
            builder.setCarrier(buildCarrier(ride));
        }

        var freeSeats = ride.getFreeSeats();
        if (freeSeats >= 0) {
            builder.setFreeSeats(freeSeats);
        }

        var ticketLimit = ride.getTicketLimit();
        if (ticketLimit != 0) {
            builder.setTicketLimit(ticketLimit);
        }

        builder.setRefundConditions(ride.getRefundConditions());

        return builder.build();
    }

    @NotNull
    private String convertSeconds(long seconds) {
        return instantToProtoWithoutTimezone(Instant.ofEpochSecond(seconds));
    }

    @NotNull
    private LegalEntity convertSupplier(@NotNull TSupplier supplier) {
        var builder = LegalEntity.newBuilder();

        builder.setRegisterType(REGISTER_TYPE_MAP.getOrDefault(supplier.getRegisterType(), RegisterType.Enum.INVALID));
        builder.setRegisterNumber(supplier.getRegisterNumber());
        consumeNonEmpty(builder::setName, supplier.getName());
        consumeNonEmpty(builder::setLegalName, supplier.getLegalName());
        consumeNonEmpty(builder::setLegalAddress, supplier.getLegalAddress());
        consumeNonEmpty(builder::setActualAddress, supplier.getActualAddress());
        consumeNonEmpty(builder::setTaxationNumber, supplier.getInn());
        consumeNonEmpty(builder::setTimetable, supplier.getTimetable());

        return builder.build();
    }

    @NotNull
    private LegalEntity convertCarrier(@NotNull TCarrier carrier) {
        var builder = LegalEntity.newBuilder();

        builder.setRegisterType(REGISTER_TYPE_MAP.getOrDefault(carrier.getRegisterType(), RegisterType.Enum.INVALID));
        builder.setRegisterNumber(carrier.getRegisterNumber());
        consumeNonEmpty(builder::setName, carrier.getName());
        consumeNonEmpty(builder::setLegalName, carrier.getLegalName());
        consumeNonEmpty(builder::setLegalAddress, carrier.getLegalAddress());
        consumeNonEmpty(builder::setActualAddress, carrier.getActualAddress());
        consumeNonEmpty(builder::setTaxationNumber, carrier.getInn());
        consumeNonEmpty(builder::setTimetable, carrier.getTimetable());

        if (carrier.getRegisterType() == ERegisterType.REGISTER_TYPE_ENTREPRENEUR && builder.getLegalName().isEmpty()) {
            builder.setLegalName(String.format(
                    "ИП %s",
                    Stream.of(carrier.getLastName(), carrier.getFirstName(), carrier.getMiddleName())
                            .filter(s -> !s.isEmpty())
                            .collect(Collectors.joining(" "))));
        }

        return builder.build();
    }

    @NotNull
    private LegalEntity buildCarrier(@NotNull TRide ride) {
        return LegalEntity.newBuilder()
                .setName(ride.getCarrierName())
                .setRegisterType(RegisterType.Enum.INVALID)
                .setLegalName(ride.getCarrierName())
                .build();
    }

    private BenefitType.Enum convertBenefit(EBenefitType benefitType) {
        return BENEFIT_TYPE_MAP.getOrDefault(benefitType, BenefitType.Enum.INVALID);
    }

    private Integer extractFirstInteger(@NotNull String s) {
        String s1 = s;
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (c < '0' || c > '9') {
                s1 = s.substring(0, i);
                break;
            }
        }
        if (s1.isEmpty()) {
            return null;
        }
        return Integer.valueOf(s1);
    }

    @NotNull
    private BookParams convertBookParams(@NotNull TBookParams bookParams, TPrice supplierFee) {
        var builder = BookParams.newBuilder();

        builder.addAllDocumentTypes(bookParams.getDocumentsList().stream().map(this::convertDocumentType)::iterator);
        builder.addAllTicketTypes(bookParams.getTicketTypesList().stream().map(tt -> convertTicketType(tt,
                supplierFee))::iterator);
        builder.addAllGenders(bookParams.getGendersList().stream().map(this::convertGender)::iterator);
        builder.addAllSeats(bookParams.getSeatsList().stream().map(TSeat::getId).sorted((a, b) -> {
            Integer aInt = extractFirstInteger(a);
            Integer bInt = extractFirstInteger(b);
            if (aInt != null && bInt != null) {
                return aInt.compareTo(bInt);
            } else {
                return a.compareTo(b);
            }
        })::iterator);
        builder.addAllPlacesMap(bookParams.getPlacesMapList().stream().map(this::convertPlace)::iterator);
        builder.addAllCitizenships(
                bookParams.getCitizenshipsList().stream()
                        .map(this::convertCitizenship)
                        .filter(Objects::nonNull)::iterator
        );

        return builder.build();
    }

    private DocumentType.Enum convertDocumentType(@NotNull TDocumentType documentType) {
        return DOCUMENT_TYPE_MAP.getOrDefault(documentType.getId(), DocumentType.Enum.invalid);
    }

    @NotNull
    private TicketType convertTicketType(@NotNull TTicketType ticketType, TPrice supplierFee) {
        var builder = TicketType.newBuilder();

        builder.setType(TICKET_TYPE_MAP.getOrDefault(ticketType.getId(), TicketTypeType.Enum.INVALID));

        var fareBuilder = Fare.newBuilder();
        var totalAmount = convertPrice(ticketType.getPrice());
        var supplierFeeAmount = ticketType.hasPartnerFee()
                ? convertPrice(ticketType.getPartnerFee())
                : convertPrice(supplierFee);
        var yandexFeeAmount = ProtoUtils.hasFields(ticketType.getYandexFee()) ?
                // FIXME buses backend should provide yandex fee (TRAVELBACK-2253)
                convertPrice(ticketType.getYandexFee()) : MoneyAmount.getDefaultInstance();

        fareBuilder.setTotal(totalAmount);
        if (totalAmount.getCurrency() == yandexFeeAmount.getCurrency() && totalAmount.getCurrency() == supplierFeeAmount.getCurrency()) {
            fareBuilder.setTicket(
                    totalAmount.toBuilder()
                            .setValue(totalAmount.getValue() - yandexFeeAmount.getValue() - supplierFeeAmount.getValue())
                            .build()
            );
        } else {
            log.error("Cannot set ticket price because of different currencies total:{} yandexFee:{} supplierFee:{}",
                    totalAmount.getCurrency(), yandexFeeAmount.getCurrency(), supplierFeeAmount.getCurrency());
        }
        fareBuilder.setSupplierFee(supplierFeeAmount);
        if (yandexFeeAmount.getCurrency() != Currency.Enum.INVALID) {
            // FIXME buses backend should provide yandex fee
            fareBuilder.setYandexFee(yandexFeeAmount);
        }

        builder.setFare(fareBuilder);

        return builder.build();
    }

    private Gender.Enum convertGender(@NotNull TGender gender) {
        return GENDER_MAP.getOrDefault(gender.getId(), Gender.Enum.invalid);
    }

    @NotNull
    private Place convertPlace(@NotNull TPlace place) {
        var builder = Place.newBuilder();

        builder.setId(place.getId());
        builder.setX(place.getX());
        builder.setY(place.getY());
        builder.setType(PLACE_TYPE_MAP.getOrDefault(place.getType(), PlaceType.Enum.INVALID));
        builder.setStatus(PLACE_STATUS_MAP.getOrDefault(place.getStatus(), PlaceStatus.Enum.INVALID));

        return builder.build();
    }

    @NotNull
    private MoneyAmount convertPrice(TPrice price) {
        var money = ProtoUtils.fromTPrice(price);
        return moneyToProto(money);
    }

    private TCountry convertCitizenship(@NotNull TCitizenship citizenship) {
        var builder = TCountry.newBuilder();

        try {
            var country = countryDataProvider.getByCodeId(citizenship.getId());
            builder.setTitle(country.getTitleDefault());
            builder.setGeoId(country.getGeoId());
            builder.setCode2(country.getCode());
        } catch (NoSuchElementException e) {
            log.warn(e.toString());
            return null;
        }

        return builder.build();
    }
}
