package ru.yandex.travel.api.services.buses;

import java.time.LocalDate;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.bus.model.BusRideHttp;
import ru.yandex.travel.bus.service.BusesHttpApiService;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;

@Service
@Slf4j
@AllArgsConstructor
public class BusesFacadeService {
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final BusesHttpApiService busesHttpApiService;
    private final Retry retryHelper;

    public enum BUS_SEARCH_RESULT_CRITERIA {
        FULL,
        MORE_AS_POSSIBLE,
        CURRENT
    }

    public CompletableFuture<Boolean> checkBusTransferExists(int geoIdFrom, int geoIdTo, LocalDate when) {
        Optional<TSettlement> settlementFrom = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoIdFrom);
        Optional<TSettlement> settlementTo = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoIdTo);
        if (settlementFrom.isEmpty() || settlementTo.isEmpty()) {
            log.error("Could not get settlements from geoId for HappyPage Transport blocks");
            return CompletableFuture.completedFuture(false);
        }
        String raspIdFrom = TrainSettlementDataProvider.Companion.prepareRaspSettlementId(settlementFrom.get().getId());
        String raspIdTo = TrainSettlementDataProvider.Companion.prepareRaspSettlementId(settlementTo.get().getId());

        try {
            return searchBusRides(
                    raspIdFrom,
                    raspIdTo,
                    when.toString(),
                    BUS_SEARCH_RESULT_CRITERIA.CURRENT,
                    ""
            ).thenApply(response -> response.length > 0);
        } catch (RuntimeException exception) {
            log.error("Exception occurred in checkBusTransferExists: ", exception);
            return CompletableFuture.completedFuture(false);
        }
    }

    public CompletableFuture<BusRideHttp[]> searchBusRides(String fromKey, String toKey, String date, BUS_SEARCH_RESULT_CRITERIA criteria, String requestId) {
        if (criteria == BUS_SEARCH_RESULT_CRITERIA.CURRENT) {
            return busesHttpApiService.search(fromKey, toKey, date, requestId).thenApply(busRsp -> busRsp.getRides());
        }

        return retryHelper.withRetry(
                "BusesFacadeService::searchBusRides",
                ignored -> busesHttpApiService.search(fromKey, toKey, date, requestId),
                new BusesSearchRetryStrategy(3)
        ).thenApply(busRsp -> {
            if (criteria == BUS_SEARCH_RESULT_CRITERIA.FULL && busRsp.isPartialResult()) {
                throw new RuntimeException("searchBusRides return partial response but expected full");
            }
            return busRsp.getRides();
        });
    }
}
