package ru.yandex.travel.api.services.cache;

import java.time.Duration;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.errorprone.annotations.CompatibleWith;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import org.checkerframework.checker.nullness.qual.NonNull;

import ru.yandex.bolts.function.Function;

public class SimpleSyncCache<K, V>  {
    private final Cache<K, V> cache;
    private final Counter getCounter;
    private final Counter putCounter;

    public SimpleSyncCache(Integer maxItemsToCache, Duration cacheDuration, String counterPrefix, String name) {
        getCounter = Metrics.counter(String.format("%s.%s.get", counterPrefix, name));
        putCounter = Metrics.counter(String.format("%s.%s.put", counterPrefix, name));
        Gauge.builder(String.format("%s.%s.hitRate", counterPrefix, name), this::getHitRate).register(Metrics.globalRegistry);
        var builder = Caffeine.newBuilder();
        if (maxItemsToCache != null) {
            builder.maximumSize(maxItemsToCache);
        }
        if (cacheDuration != null) {
            builder.expireAfterWrite(cacheDuration);
        }
        cache = builder.build();
    }

    public void invalidate(@NonNull @CompatibleWith("K") Object key) {
        cache.invalidate(key);
    }

    public void put(K key, V value) {
        putCounter.increment();
        cache.put(key, value);
    }

    public V get(K key) {
        getCounter.increment();
        return cache.getIfPresent(key);
    }

    public V getOrCompute(K key, Function<? super K, ? extends V> mappingFunction) {
        getCounter.increment();
        return cache.get(key, (k) -> {
            putCounter.increment();
            return mappingFunction.apply(k);
        });
    }

    public double getHitRate() {
        return cache.stats().hitRate();
    }
}
