package ru.yandex.travel.api.services.common;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.spec.Currency;
import ru.yandex.travel.api.spec.MoneyAmount;
import ru.yandex.travel.api.spec.OrderRefundState;
import ru.yandex.travel.api.spec.OrderRefundType;
import ru.yandex.travel.api.spec.RefundInfo;
import ru.yandex.travel.api.spec.RefundPartContext;
import ru.yandex.travel.api.spec.RefundPartInfo;
import ru.yandex.travel.api.spec.RefundPartState;
import ru.yandex.travel.api.spec.RefundPartType;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TRefundPartInfo;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE_TIME;

public class ApiSpecProtoUtils {
    private static final Map<CurrencyUnit, Currency.Enum> CURRENCY_MAP = Map.of(
            ProtoCurrencyUnit.RUB, Currency.Enum.RUB,
            ProtoCurrencyUnit.USD, Currency.Enum.USD,
            ProtoCurrencyUnit.EUR, Currency.Enum.EUR
    );
    private static final Map<ERefundPartState, RefundPartState.Enum> REFUND_PART_STATE_MAP = Map.of(
            ERefundPartState.RPS_DISABLED, RefundPartState.Enum.DISABLED,
            ERefundPartState.RPS_ENABLED, RefundPartState.Enum.ENABLED,
            ERefundPartState.RPS_REFUNDED, RefundPartState.Enum.REFUNDED,
            ERefundPartState.RPS_DEPENDENT, RefundPartState.Enum.DEPENDENT,
            ERefundPartState.RPS_OFFLINE_ENABLED, RefundPartState.Enum.OFFLINE_ENABLED
    );
    private static final Map<ERefundPartType, RefundPartType.Enum> REFUND_PART_TYPE_MAP = Map.of(
            ERefundPartType.RPT_ORDER, RefundPartType.Enum.ORDER,
            ERefundPartType.RPT_SERVICE, RefundPartType.Enum.SERVICE,
            ERefundPartType.RPT_SERVICE_PART, RefundPartType.Enum.SERVICE_PART
    );
    private static final Map<EOrderRefundType, OrderRefundType.Enum> REFUND_TYPE_MAP = Map.of(
            EOrderRefundType.RT_GENERIC_USER_REFUND, OrderRefundType.Enum.USER_REFUND,
            EOrderRefundType.RT_TRAIN_USER_REFUND, OrderRefundType.Enum.USER_REFUND,
            EOrderRefundType.RT_TRAIN_OFFICE_REFUND, OrderRefundType.Enum.TRAIN_OFFICE_REFUND,
            EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN, OrderRefundType.Enum.TRAIN_INSURANCE_AUTO_RETURN
    );
    private static final Map<EOrderRefundState, OrderRefundState.Enum> REFUND_STATE_MAP = Map.of(
            EOrderRefundState.RS_WAITING_SERVICE_REFUND, OrderRefundState.Enum.IN_PROGRESS,
            EOrderRefundState.RS_WAITING_INVOICE_REFUND, OrderRefundState.Enum.IN_PROGRESS,
            EOrderRefundState.RS_FAILED, OrderRefundState.Enum.FAILED,
            EOrderRefundState.RS_REFUNDED, OrderRefundState.Enum.REFUNDED
    );

    public static String instantToProtoWithoutTimezone(Instant datetime) {
        if (datetime == null) {
            return null;
        }
        return ISO_LOCAL_DATE_TIME.withZone(ZoneOffset.UTC).format(datetime);
    }

    public static String localDateToProto(LocalDate date) {
        if (date == null) {
            return null;
        }
        return date.toString();
    }

    /**
    * Obtains an instance of LocalDate from a text string such as 2007-12-03.
    * Params: date – the text to parse such as "2007-12-03", or null or ""
    * Returns: the parsed local date, or null
     */
    public static LocalDate localDateFromProto(String date) {
        if (Strings.isNullOrEmpty(date)) {
            return null;
        }
        return LocalDate.parse(date);
    }

    public static <T> void consumeNotNull(Consumer<T> consumer, T value) {
        if (value != null) {
            consumer.accept(value);
        }
    }

    public static void consumeNonEmpty(Consumer<String> consumer, String value) {
        if (!Strings.isNullOrEmpty(value)) {
            consumer.accept(value);
        }
    }

    public static MoneyAmount moneyToProto(Money money) {
        if (money == null) {
            return null;
        }
        var mappedCurrency = requireMapEnum(CURRENCY_MAP, money.getCurrency(), "CurrencyUnit");
        return MoneyAmount.newBuilder()
                .setValue(money.getNumber().floatValue())
                .setCurrency(mappedCurrency)
                .build();
    }

    public static <S, R> R requireMapEnum(Map<S, R> map, S source, String enumName) {
        var result = map.get(source);
        Preconditions.checkArgument(result != null, "Unknown %s value %s", enumName, source.toString());
        return result;
    }

    public static RefundPartInfo refundPartInfoToProto(TRefundPartInfo source, ApiTokenEncrypter apiTokenEncrypter) {
        if (source == null) {
            return null;
        }
        var rp = RefundPartInfo.newBuilder();
        rp.setState(requireMapEnum(REFUND_PART_STATE_MAP, source.getState(), "RefundPartState"));
        rp.setType(requireMapEnum(REFUND_PART_TYPE_MAP, source.getType(), "RefundPartType"));
        rp.setContext(RefundPartContext.newBuilder().setInfo(source.getContext()).build());
        if (source.hasRefund()) {
            var refund = RefundInfo.newBuilder();
            refund.setId(source.getRefund().getId());
            refund.setType(requireMapEnum(REFUND_TYPE_MAP, source.getRefund().getRefundType(), "RefundType"));
            refund.setState(requireMapEnum(REFUND_STATE_MAP, source.getRefund().getState(), "RefundState"));
            if (source.getRefund().hasRefundAmount()) {
                refund.setRefundAmount(moneyToProto(ProtoUtils.fromTPrice(source.getRefund().getRefundAmount())));
            }
            if (source.getRefund().hasRefundBlankToken()) {
                refund.setRefundBlankToken(apiTokenEncrypter.toDownloadBlankToken(source.getRefund().getRefundBlankToken()));
            }
            refund.addAllPaymentRefundReceiptUrls(source.getRefund().getPaymentRefundReceiptUrlsList().stream()
                    .map(ApiSpecProtoUtils::convertReceiptUrlToPdf).collect(Collectors.toList()));
            rp.setRefund(refund);
        }
        return rp.build();
    }

    public static String convertReceiptUrlToPdf(String receiptUrl) {
        return receiptUrl.replace("mode=mobile", "mode=pdf");
    }
}
