package ru.yandex.travel.api.services.common;

import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.config.common.PdfDownloadConfigurationProperties;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;

@Service
@EnableConfigurationProperties(PdfDownloadConfigurationProperties.class)
public class PdfDownloadService {
    private final AsyncHttpClientWrapper client;
    private final PdfDownloadConfigurationProperties config;

    public PdfDownloadService(@Qualifier("pdfAhcWrapper") AsyncHttpClientWrapper client,
                              PdfDownloadConfigurationProperties config) {
        this.client = client;
        this.config = config;
    }

    public CompletableFuture<ResponseEntity<byte[]>> downloadDocumentAsBytesAsync(String documentUrl) {
        if (Strings.isNullOrEmpty(documentUrl)) {
            return CompletableFuture.completedFuture(ResponseEntity.status(HttpStatus.NOT_FOUND).build());
        }
        RequestBuilder builder = new RequestBuilder()
                .setMethod("GET")
                .setRequestTimeout(Math.toIntExact(config.getRequestTimeout().toMillis()))
                .setReadTimeout(Math.toIntExact(config.getReadTimeout().toMillis()))
                .setUrl(documentUrl);
        return client.executeRequest(builder, "download")
                .thenApply(response -> {
                    if (response.getStatusCode() == 200) {
                        return ResponseEntity.status(HttpStatus.OK)
                                .contentType(MediaType.APPLICATION_PDF)
                                .body(response.getResponseBodyAsBytes());
                    } else {
                        return ResponseEntity.status(response.getStatusCode()).build();
                    }
                });
    }
}
