package ru.yandex.travel.api.services.common;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

@Service
@Slf4j
public class PhoneCountryCodesService {
    private static final String COUNTRY_CODES_FILE = "utils/phone_country_codes.txt";

    private final Set<Integer> countryCodes = ImmutableSet.copyOf(loadCountryCodes(COUNTRY_CODES_FILE));

    @SuppressWarnings("SameParameterValue")
    private static Set<Integer> loadCountryCodes(String file) {
        InputStream input = PhoneCountryCodesService.class.getClassLoader().getResourceAsStream(file);
        Preconditions.checkNotNull(input, "No input file; path {}", file);

        try (BufferedReader reader = new BufferedReader(new InputStreamReader(input))) {
            Set<Integer> codes = new HashSet<>();
            for (String line; (line = reader.readLine()) != null; ) {
                // # codes
                // +7: RU, KZ
                if (Strings.isNullOrEmpty(line) || line.startsWith("#")) {
                    continue;
                }
                String code = line.contains(": ") ? line.substring(0, line.indexOf(": ")) : line;
                codes.add(Integer.parseInt(code.replace("+", "")));
            }
            log.info("Loaded {} country codes from {}", codes.size(), file);
            return codes;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @param normalizedPhone - not empty, digits only, no leading zeroes, russian local country code 8 replaced with 7
     */
    public Integer getCountryCodeSafe(String normalizedPhone) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(normalizedPhone) && normalizedPhone.matches("\\d+"),
                "Not a normalized phone: {}", normalizedPhone);
        for (int i = 1; i < normalizedPhone.length() && i < 10; i++) {
            Integer candidate = Integer.valueOf(normalizedPhone.substring(0, i));
            if (countryCodes.contains(candidate)) {
                return candidate;
            }
        }
        int codeLength = Math.min(3, normalizedPhone.length());
        return Integer.valueOf(normalizedPhone.substring(0, codeLength));
    }
}
