package ru.yandex.travel.api.services.common;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.Objects;

import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Mir2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.PromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Taxi2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.WhiteLabelCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.YandexEdaCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.YandexPlusCampaignDto;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.WhiteLabelPromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.YandexPlusPromoCampaign;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.EWhiteLabelEligibility;
import ru.yandex.travel.hotels.proto.EYandexEdaEligibility;

public class PromoCampaignUtils {
    public static PromoCampaignsDto createPromoCampaignsDto(PromoCampaignsInfo campaigns) {
        PromoCampaignsDto campaignsDto = new PromoCampaignsDto();
        if (campaigns.getTaxi2020() != null) {
            campaignsDto.setTaxi2020(Taxi2020PromoCampaignDto.builder()
                    .eligible(campaigns.getTaxi2020().isEligible())
                    .build());
        }
        if (campaigns.getMir2020() != null) {
            var builder = Mir2020PromoCampaignDto.builder();
            if (campaigns.getMir2020().getEligibility() == EMirEligibility.ME_ELIGIBLE) {
                builder.eligible(true)
                        .cashbackAmount(campaigns.getMir2020().getCashbackAmount());
                if (campaigns.getMir2020().getDisplayAmount() == null || campaigns.getMir2020().getDisplayAmount()) {
                    builder.cashbackAmountString(formatRublePrice(campaigns.getMir2020().getCashbackAmount()));
                } else {
                    builder.cashbackAmountString(formatRate(campaigns.getMir2020().getCashbackRate()));
                }
            } else {
                builder.eligible(false);
            }
            campaignsDto.setMir2020(builder.build());
        }
        if (campaigns.getYandexPlus() != null) {
            YandexPlusPromoCampaign yandexPlus = campaigns.getYandexPlus();
            campaignsDto.setYandexPlus(YandexPlusCampaignDto.builder()
                    .eligible(yandexPlus.getEligible() == Boolean.TRUE)
                    .points(Objects.requireNonNullElse(yandexPlus.getPoints(), 0))
                    .withdrawPoints(yandexPlus.getWithdrawPoints())
                    .build());
        }
        if (campaigns.getYandexEda() != null) {
            var builder = YandexEdaCampaignDto.builder();
            if (campaigns.getYandexEda().getEligible() == EYandexEdaEligibility.YEE_ELIGIBLE) {
                builder.eligible(true)
                        .data(YandexEdaCampaignDto.YandexEdaPromocodePayload.builder()
                                .numberOfPromocodes(campaigns.getYandexEda().getData().getNumberOfPromocodes())
                                .promocodeCost(campaigns.getYandexEda().getData().getPromocodeCost())
                                .firstSendDate(campaigns.getYandexEda().getData().getFirstSendDate())
                                .lastSendDate(campaigns.getYandexEda().getData().getLastSendDate())
                        .build());
            } else {
                builder.eligible(false);
            }
            campaignsDto.setYandexEda(builder.build());
        }
        if (campaigns.getWhiteLabel() != null) {
            WhiteLabelPromoCampaign whiteLabel = campaigns.getWhiteLabel();
            var builder = WhiteLabelCampaignDto.builder();
            boolean eligible = whiteLabel.getEligible() == EWhiteLabelEligibility.WLE_ELIGIBLE;
            builder.eligible(eligible);
            if (eligible)
                builder.points(WhiteLabelCampaignDto.WhiteLabelPoints.builder()
                        .amount(whiteLabel.getPoints().getAmount())
                        .pointsType(whiteLabel.getPoints().getPointsType())
                        .pointsName(whiteLabel.getPointsLinguistics().getNameForNumeralNominative())
                        .build());
            campaignsDto.setWhiteLabel(builder.build());
        }
        return campaignsDto;
    }

    private static String formatRate(BigDecimal rate) {
        var format = new DecimalFormat("0%");
        return format.format(rate);
    }

    private static String formatRublePrice(int price) {
        DecimalFormat formatter = (DecimalFormat) NumberFormat.getInstance();
        DecimalFormatSymbols symbols = formatter.getDecimalFormatSymbols();
        symbols.setGroupingSeparator(' ');
        formatter.setDecimalFormatSymbols(symbols);
        return formatter.format(price) + " ₽";
    }
}
