package ru.yandex.travel.api.services.common;

import java.time.Duration;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.commons.retry.RetryStrategy;
import ru.yandex.travel.commons.retry.RetryStrategyBuilder;
import ru.yandex.travel.orders.client.HAGrpcException;

/**
 * Retry strategy class is parameterized by the type of Function used inside Retry
 * so we use this helper to ease the creation of ad-hoc (parameter of the Retry call) RetryStrategy
 */
public class RetryStrategyExceptionHelpers {
    private static final int DEFAULT_GRPC_RETRY_TIMES = 3;
    private static final Duration DEFAULT_GRPC_RETRY_TIMEOUT = Duration.ofMillis(100);

    public RetryStrategyExceptionHelpers() {
    }

    public static <T> RetryStrategy<T> defaultRetryStrategy() {
        return defaultStatusUnavailableRetryStrategy();
    }

    public static <T> RetryStrategy<T> defaultStatusUnavailableRetryStrategy() {
        return buildDefaultRetryStrategy(DEFAULT_GRPC_RETRY_TIMES, DEFAULT_GRPC_RETRY_TIMEOUT);
    }

    @SuppressWarnings("RedundantIfStatement")
    public static <T> RetryStrategy<T> buildDefaultRetryStrategy(int numRetries, Duration timeout) {
        return new RetryStrategyBuilder<T>()
                .retryOnException(e -> {
                    if (e instanceof StatusRuntimeException) {
                        StatusRuntimeException casted = (StatusRuntimeException) e;
                        return casted.getStatus().getCode() == Status.Code.UNAVAILABLE;
                    } else if (e instanceof HAGrpcException) {
                        return true;
                    } else if (e instanceof HttpApiRetryableException) {
                        return true;
                    } else {
                        return false;
                    }
                })
                .setTimeout(timeout)
                .setNumRetries(numRetries)
                .build();
    }

}
