package ru.yandex.travel.api.services.crosslinks;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlock;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlockData;
import ru.yandex.travel.api.models.hotels.HotelWithMinPrice;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.min_prices.RegionMinPricesService;
import ru.yandex.travel.api.services.hotels.region_hotel_count.RegionHotelCountService;
import ru.yandex.travel.api.services.hotels.region_images.RegionImagesService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionHotelsSearcher;
import ru.yandex.travel.api.services.hotels.static_pages.RegionHotelsViaGeocounterSearcher;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.proto.region_pages.EGeoSearchSortType;

@Component
@RequiredArgsConstructor
@Slf4j
public class CrosslinksService implements ICrosslinksService {
    public final static int HOTEL_CROSSLINKS_DEFAULT_LIMIT = 7;

    private final static String CROSS_SALE_REGION_IMAGE_SIZE = "region-desktop";

    private final GeoBase geoBase;
    private final RegionImagesService regionImagesService;
    private final RegionHotelsViaGeocounterSearcher regionHotelsViaGeocounterSearcher;
    private final RegionHotelCountService regionHotelCountService;
    private final RegionMinPricesService regionMinPricesService;
    private final RegionsService regionsService;

    public CompletableFuture<CrosslinksHotelsBlock> getHotelsBlock(int geoId, String domain,
                                                                   CommonHttpHeaders commonHttpHeaders,
                                                                   UserCredentials userCredentials) {
        return getHotelsBlock(geoId, HOTEL_CROSSLINKS_DEFAULT_LIMIT, domain, commonHttpHeaders, userCredentials);
    }

    public CompletableFuture<CrosslinksHotelsBlock> getHotelsBlock(int geoId, int hotelsLimit, String domain,
                                                                   CommonHttpHeaders commonHttpHeaders,
                                                                   UserCredentials userCredentials) {

        return getHotelsBlockData(geoId, hotelsLimit, domain, commonHttpHeaders, userCredentials)
                .thenApply(CrosslinksHotelsBlock::new);
    }

    public CompletableFuture<CrosslinksHotelsBlockData> getHotelsBlockData(
            int geoId, String domain,
            CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        return getHotelsBlockData(geoId, HOTEL_CROSSLINKS_DEFAULT_LIMIT, domain, commonHttpHeaders, userCredentials);
    }

    public CompletableFuture<CrosslinksHotelsBlockData> getHotelsBlockData(
            int geoId, int hotelsLimit, String domain,
            CommonHttpHeaders commonHttpHeaders, UserCredentials userCredentials) {
        var requestData = new RegionSearchHotelsRequestData();
        requestData.setGeoId(geoId);
        requestData.setLimit(hotelsLimit);
        requestData.setSortType(EGeoSearchSortType.RELEVANT_FIRST);

        return regionHotelsViaGeocounterSearcher.searchHotelsWithMinPrices(geoId, requestData, commonHttpHeaders, userCredentials, "CrossLinks")
                .thenApply(result -> prepareHotelsBlockData(geoId, domain, result));
    }

    private CrosslinksHotelsBlockData prepareHotelsBlockData(
            int geoId, String domain,
            RegionHotelsViaGeocounterSearcher.SearchResultWithMinPrices searchResultWithMinPrices) {
        var blockData = new CrosslinksHotelsBlockData();

        blockData.setHasData(searchResultWithMinPrices.getTotalHotelsFound() > 0);

        if (blockData.isHasData()) {
            var region = regionsService.getRegion(geoId, domain, "ru");
            blockData.setRegion(region);

            var priceByRegion = regionMinPricesService.getPriceByGeoId(geoId);
            if (priceByRegion.isPresent()) {
                blockData.setMinPriceInRegion(new Price(priceByRegion.getAsInt(), Price.Currency.RUB));
            } else {
                blockData.setMinPriceInRegion(searchResultWithMinPrices.getHotelsWithMinPrice().stream()
                        .map(RegionHotelsSearcher.HotelWithMinPrice::getMinPrice)
                        .min(Comparator.comparing(Price::getValue))
                        .orElse(new Price(0, Price.Currency.RUB)));
            }

            var bbox = HotelsPortalUtils.getBBoxByHotels(geoId, domain, geoBase,
                    searchResultWithMinPrices
                            .getHotelsWithMinPrice()
                            .stream()
                            .map(RegionHotelsSearcher.HotelWithMinPrice::getHotel)
                            .collect(Collectors.toList()));
            blockData.setBboxAsString(bbox);
            blockData.setBboxAsStruct(List.of(bbox.getLeftDown(), bbox.getUpRight()));

            blockData.setHotels(searchResultWithMinPrices.getHotelsWithMinPrice().stream()
                    .map(x -> new HotelWithMinPrice(x.getHotel(), x.getMinPrice(), x.getBadges(), x.getHasOfferYandexPlus()))
                    .collect(Collectors.toUnmodifiableList()));

            blockData.setTotalHotelCount(regionHotelCountService.getHotelCount(geoId)
                    .orElse(searchResultWithMinPrices.getTotalHotelsFound())
            );

            blockData.setRegionImageUrl(regionImagesService.getImageUsingTreeWithSize(geoId, CROSS_SALE_REGION_IMAGE_SIZE));
        }

        return blockData;
    }
}
