package ru.yandex.travel.api.services.dictionaries;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.caffinitas.ohc.CacheSerializer;
import org.caffinitas.ohc.Eviction;
import org.caffinitas.ohc.OHCache;
import org.caffinitas.ohc.OHCacheBuilder;

@Slf4j
public class DictionaryUtils {
    public static <T extends Message> void readDictionary(File file, IoFunction<InputStream, T> parser, Consumer<T> consumer) {
        try (FileInputStream in = new FileInputStream(file)) {
            T object;
            while ((object = parser.apply(in)) != null) {
                consumer.accept(object);
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to read the input file: " + file.getAbsolutePath(), e);
        }
    }

    public static <T extends Message> void readDictionaryCoded(File file, IoFunction<CodedInputStream, T> parser,
                                                               Consumer<T> consumer) {
        try (FileInputStream in = new FileInputStream(file)) {
            CodedInputStream cis = CodedInputStream.newInstance(in);
            T object;
            while (!cis.isAtEnd()) {
                object = parser.apply(cis);
                consumer.accept(object);
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to read the input file: " + file.getAbsolutePath(), e);
        }
    }

    public static <T extends Message> IoFunction<CodedInputStream, T> parseIntSizeMessage(IoFunction<byte[], T> parser) {
        return cis -> {
            int size = cis.readFixed32();
            return parser.apply(cis.readRawBytes(size));
        };
    }

    public static <T extends Message> IoFunction<CodedInputStream, T> parseBufferMessage(IoFunction<ByteBuffer, T> parser) {
        return cis -> {
            var size = cis.readFixed32();
            var bytes = cis.readRawBytes(size);
            return parser.apply(ByteBuffer.wrap(bytes));
        };
    }

    public static <K, V extends Message> OHCache<K, V> createDefaultOhCacheFor(int capacityBytes,
                                                                        CacheSerializer<K> ohcKeySerializer,
                                                                        CacheSerializer<V> ohcValueSerializer) {
        return OHCacheBuilder.<K, V>newBuilder()
                .capacity(capacityBytes)
                .keySerializer(ohcKeySerializer)
                .valueSerializer(ohcValueSerializer)
                .eviction(Eviction.NONE)
                .timeouts(false)
                .build();
    }

    private static <K, V extends Message> void addToCache(
            OHCache<K, V> cache, V object, Function<V, K> getKey, Function<V, Boolean> filter) {
        if (filter == null || filter.apply(object)) {
            var key = getKey.apply(object);
            boolean unique = !cache.containsKey(key);
            Preconditions.checkArgument(unique, "Key %s duplicated", key);
            boolean inserted = cache.put(key, object);
            Preconditions.checkArgument(inserted,
                    "Failed to insert cache record; cache.size=%s", cache.size());
        }
    }

    public static <K, V extends Message> OHCache<K, V> createCacheFromFile
            (DictionaryProperties properties, CacheSerializer<K> keySerializer,
             DictionaryUtils.IoFunction<ByteBuffer, V> bufferParser, Function<V, K> getKey) {
        return createCacheFromFile(properties, keySerializer, bufferParser, getKey, null);
    }

    public static <K, V extends Message> OHCache<K, V> createCacheFromFile
            (DictionaryProperties properties, CacheSerializer<K> keySerializer,
             DictionaryUtils.IoFunction<ByteBuffer, V> bufferParser, Function<V, K> getKey,
             Function<V, Boolean> filter) {
        log.info("Loading data cache with the config: {}", properties);
        File dataFile = new File(properties.getDataFile());
        Preconditions.checkArgument(dataFile.exists(), "No such file: %s", dataFile.getAbsolutePath());
        var capacityBytes = (int) (dataFile.length() * properties.getCapacityCoefficient());
        var valueSerializer = new DictionaryUtils.ProtoMessageSerializer<>(bufferParser);
        OHCache<K, V> cache = DictionaryUtils.createDefaultOhCacheFor(capacityBytes, keySerializer, valueSerializer);
        DictionaryUtils.readDictionaryCoded(
                dataFile, DictionaryUtils.parseBufferMessage(bufferParser),
                obj -> addToCache(cache, obj, getKey, filter));
        log.info("Initialized with {} objects", cache.size());
        return cache;
    }

    public static <K, V extends Message> OHCache<K, V> createEmptyCache(
            CacheSerializer<K> keySerializer, DictionaryUtils.IoFunction<ByteBuffer, V> bufferParser) {
        log.warn("Creating an empty dictionary");
        var valueSerializer = new DictionaryUtils.ProtoMessageSerializer<>(bufferParser);
        int emptySize = 1;
        return DictionaryUtils.createDefaultOhCacheFor(emptySize, keySerializer, valueSerializer);
    }

    public static <K, V extends Message> OHCache<K, V> createCacheFromIterator(
            int capacityBytes, CacheSerializer<K> keySerializer,
            DictionaryUtils.IoFunction<ByteBuffer, V> bufferParser, Iterator<V> content,
            Function<V, K> getKey) {
        return createCacheFromIterator(capacityBytes, keySerializer, bufferParser, content, getKey, null);
    }

    public static <K, V extends Message> OHCache<K, V> createCacheFromIterator(
            int capacityBytes, CacheSerializer<K> keySerializer,
            DictionaryUtils.IoFunction<ByteBuffer, V> bufferParser, Iterator<V> content,
            Function<V, K> getKey, Function<V, Boolean> filter) {
        log.info("Loading data cache from map, size {}", capacityBytes);
        var valueSerializer = new DictionaryUtils.ProtoMessageSerializer<>(bufferParser);
        OHCache<K, V> cache = DictionaryUtils.createDefaultOhCacheFor(capacityBytes, keySerializer, valueSerializer);
        content.forEachRemaining(obj -> addToCache(cache, obj, getKey, filter));
        log.info("Initialized with {} objects", cache.size());
        return cache;
    }

    public interface IoFunction<T, R> {
        R apply(T value) throws IOException;
    }

    public static final class LongSerializer implements CacheSerializer<Long> {
        @Override
        public void serialize(Long value, ByteBuffer buf) {
            buf.putLong(value);
        }

        @Override
        public Long deserialize(ByteBuffer buf) {
            return buf.getLong();
        }

        @Override
        public int serializedSize(Long value) {
            return 8;
        }
    }

    public static final class IntSerializer implements CacheSerializer<Integer> {
        @Override
        public void serialize(Integer value, ByteBuffer buf) {
            buf.putInt(value);
        }

        @Override
        public Integer deserialize(ByteBuffer buf) {
            return buf.getInt();
        }

        @Override
        public int serializedSize(Integer value) {
            return 4;
        }
    }

    public static final class StringSerializer implements CacheSerializer<String> {
        @Override
        public void serialize(String value, ByteBuffer buf) {
            buf.put(value.getBytes());
        }

        @Override
        public String deserialize(ByteBuffer buf) {
            return buf.toString();
        }

        @Override
        public int serializedSize(String value) {
            return value.getBytes().length;
        }
    }

    @AllArgsConstructor
    public static final class ProtoMessageSerializer<T extends Message> implements CacheSerializer<T> {
        private final IoFunction<ByteBuffer, T> byteBufferDeserializer;

        @Override
        public void serialize(T value, ByteBuffer buf) {
            buf.put(value.toByteArray());
        }

        @Override
        public T deserialize(ByteBuffer buf) {
            try {
                return byteBufferDeserializer.apply(buf);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public int serializedSize(T value) {
            return value.getSerializedSize();
        }
    }
}
