package ru.yandex.travel.api.services.dictionaries.avia;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.caffinitas.ohc.OHCache;

import ru.yandex.travel.api.services.dictionaries.DictionaryProperties;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.LongSerializer;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.ProtoMessageSerializer;
import ru.yandex.travel.dicts.avia.TAirline;

@Slf4j
public class AviaAirlineDictionary implements AutoCloseable {
    private static final String AEROFLOT_IATA_CODE = "SU";
    private static final int AEROFLOT_ID = 26;

    private final OHCache<Long, TAirline> cache;
    private final Map<String, Long> iataCodeIndex;

    public AviaAirlineDictionary(DictionaryProperties properties) {
        log.info("Loading data cache with the config: {}", properties);
        File dataFile = new File(properties.getDataFile());
        Preconditions.checkArgument(dataFile.exists(), "No such file: %s", dataFile.getAbsolutePath());
        ProtoMessageSerializer<TAirline> valueSerializer = new ProtoMessageSerializer<>(TAirline::parseFrom);
        int capacityBytes = (int) (dataFile.length() * properties.getCapacityCoefficient());
        cache = DictionaryUtils.createDefaultOhCacheFor(capacityBytes, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
        DictionaryUtils.readDictionary(dataFile, TAirline::parseDelimitedFrom, obj -> {
            boolean inserted = cache.put(obj.getId(), obj);
            Preconditions.checkArgument(inserted, "Failed to insert cache record; cache.size=%s", cache.size());
            if (!Strings.isNullOrEmpty(obj.getIataCode())) {
                Long prevId = iataCodeIndex.put(obj.getIataCode(), obj.getId());
                if (prevId != null) {
                    log.warn("IATA code '{}' collision for object ids: [{}, {}]", obj.getIataCode(), prevId, obj.getId());
                    // Aeroflot & Aurora share the same code
                    if (AEROFLOT_IATA_CODE.equals(obj.getIataCode()) && prevId == AEROFLOT_ID) {
                        log.warn("Forcing IATA code mapping for '{}': {} will be used over {}", obj.getIataCode(), prevId, obj.getId());
                        iataCodeIndex.put(obj.getIataCode(), prevId);
                    }
                }
            }
        });
        log.info("Initialized with {} objects", cache.size());
    }

    private AviaAirlineDictionary() {
        log.warn("Creating an empty dictionary");
        ProtoMessageSerializer<TAirline> valueSerializer = new ProtoMessageSerializer<>(TAirline::parseFrom);
        int emptySize = 1;
        cache = DictionaryUtils.createDefaultOhCacheFor(emptySize, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
    }

    public static AviaAirlineDictionary createEmpty() {
        return new AviaAirlineDictionary();
    }

    public TAirline getById(Long id) {
        TAirline airline = cache.get(id);
        if (airline == null) {
            throw new RuntimeException("No such airline: " + id);
        }
        return airline;
    }

    public TAirline getByIataCode(String iataCode) {
        Long id = iataCodeIndex.get(iataCode);
        if (id == null) {
            throw new RuntimeException("No such airline: " + iataCode);
        }
        return getById(id);
    }

    public boolean hasIataCode(String iataCode) {
        return iataCodeIndex.get(iataCode) != null;
    }

    @Override
    public void close() throws IOException {
        cache.close();
    }
}
