package ru.yandex.travel.api.services.dictionaries.avia;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.caffinitas.ohc.OHCache;

import ru.yandex.travel.api.services.dictionaries.DictionaryProperties;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.LongSerializer;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.ProtoMessageSerializer;
import ru.yandex.travel.dicts.avia.TAirport;

@Slf4j
public class AviaAirportDictionary implements AutoCloseable {
    private final OHCache<Long, TAirport> cache;
    private final Map<String, Long> iataCodeIndex;

    public AviaAirportDictionary(DictionaryProperties properties) {
        log.info("Loading data cache with the config: {}", properties);
        File dataFile = new File(properties.getDataFile());
        Preconditions.checkArgument(dataFile.exists(), "No such file: %s", dataFile.getAbsolutePath());
        ProtoMessageSerializer<TAirport> valueSerializer = new ProtoMessageSerializer<>(TAirport::parseFrom);
        int capacityBytes = (int) (dataFile.length() * properties.getCapacityCoefficient());
        cache = DictionaryUtils.createDefaultOhCacheFor(capacityBytes, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
        DictionaryUtils.readDictionary(dataFile, TAirport::parseDelimitedFrom, obj -> {
            boolean inserted = cache.put(obj.getId(), obj);
            Preconditions.checkArgument(inserted, "Failed to insert cache record; cache.size=%s", cache.size());
            if (!Strings.isNullOrEmpty(obj.getIataCode())) {
                Long prevId = iataCodeIndex.put(obj.getIataCode(), obj.getId());
                if (prevId != null) {
                    log.warn("Iata code '{}' collision for object ids: [{}, {}]", obj.getIataCode(), prevId, obj.getId());
                }
            }
        });
        log.info("Initialized with {} objects", cache.size());
    }

    private AviaAirportDictionary() {
        log.warn("Creating an empty dictionary");
        ProtoMessageSerializer<TAirport> valueSerializer = new ProtoMessageSerializer<>(TAirport::parseFrom);
        int emptySize = 1;
        cache = DictionaryUtils.createDefaultOhCacheFor(emptySize, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
    }

    public static AviaAirportDictionary createEmpty() {
        return new AviaAirportDictionary();
    }

    public TAirport getById(Long id) {
        TAirport airport = cache.get(id);
        if (airport == null) {
            throw new RuntimeException("No such airport: " + id);
        }
        return airport;
    }

    public TAirport getByIataCode(String iataCode) {
        Long id = iataCodeIndex.get(iataCode);
        if (id == null) {
            throw new RuntimeException("No such airport: " + iataCode);
        }
        return getById(id);
    }

    public boolean hasIataCode(String iataCode){
        return iataCodeIndex.get(iataCode)!=null;
    }

    @Override
    public void close() throws IOException {
        cache.close();
    }
}
