package ru.yandex.travel.api.services.dictionaries.avia;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.caffinitas.ohc.OHCache;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.travel.api.services.dictionaries.DictionaryProperties;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.LongSerializer;
import ru.yandex.travel.api.services.dictionaries.DictionaryUtils.ProtoMessageSerializer;
import ru.yandex.travel.dicts.avia.TSettlement;

@Slf4j
public class AviaSettlementDictionary implements AutoCloseable {
    private final OHCache<Long, TSettlement> cache;
    private final Map<String, Long> iataCodeIndex;
    private final Map<String, Long> sirenaCodeIndex;

    @Autowired
    public AviaSettlementDictionary(DictionaryProperties properties) {
        log.info("Loading data cache with the config: {}", properties);
        File dataFile = new File(properties.getDataFile());
        Preconditions.checkArgument(dataFile.exists(), "No such file: %s", dataFile.getAbsolutePath());
        ProtoMessageSerializer<TSettlement> valueSerializer = new ProtoMessageSerializer<>(TSettlement::parseFrom);
        int capacityBytes = (int) (dataFile.length() * properties.getCapacityCoefficient());
        cache = DictionaryUtils.createDefaultOhCacheFor(capacityBytes, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
        sirenaCodeIndex = new HashMap<>();
        DictionaryUtils.readDictionary(dataFile, TSettlement::parseDelimitedFrom, obj -> {
            boolean inserted = cache.put(obj.getId(), obj);
            Preconditions.checkArgument(inserted, "Failed to insert cache record; cache.size=%s", cache.size());
            if (!Strings.isNullOrEmpty(obj.getIataCode())) {
                Long prevId = iataCodeIndex.put(obj.getIataCode(), obj.getId());
                if (prevId != null) {
                    log.warn("Iata code '{}' collision for object ids: [{}, {}]", obj.getIataCode(), prevId,
                            obj.getId());
                }
            }
            if (!Strings.isNullOrEmpty(obj.getSirenaID())) {
                Long prevId = sirenaCodeIndex.put(obj.getSirenaID(), obj.getId());
                if (prevId != null) {
                    log.warn("Sirena code '{}' collision for object ids: [{}, {}]", obj.getSirenaID(), prevId,
                            obj.getId());
                }
            }
        });
        log.info("Initialized with {} objects", cache.size());
    }

    private AviaSettlementDictionary() {
        log.warn("Creating an empty dictionary");
        ProtoMessageSerializer<TSettlement> valueSerializer = new ProtoMessageSerializer<>(TSettlement::parseFrom);
        int emptySize = 1;
        cache = DictionaryUtils.createDefaultOhCacheFor(emptySize, new LongSerializer(), valueSerializer);
        iataCodeIndex = new HashMap<>();
        sirenaCodeIndex = new HashMap<>();
    }

    public static AviaSettlementDictionary createEmpty() {
        return new AviaSettlementDictionary();
    }

    public TSettlement getById(Long id) {
        TSettlement settlement = cache.get(id);
        if (settlement == null) {
            throw new RuntimeException("No such settlement: " + id);
        }
        return settlement;
    }

    public TSettlement getByCode(String code) {
        Long id = iataCodeIndex.get(code);
        if (id != null) {
            return getById(id);
        }
        id = sirenaCodeIndex.get(code);
        if (id != null) {
            return getById(id);
        }
        throw new RuntimeException("No such settlement: " + code);
    }

    public boolean hasCode(String code){
        return iataCodeIndex.get(code)!=null || sirenaCodeIndex.get(code)!=null;
    }

    public boolean hasId(long id){
        return cache.get(id) != null;
    }

    @Override
    public void close() throws IOException {
        cache.close();
    }
}
