package ru.yandex.travel.api.services.dictionaries.country

import org.apache.lucene.document.Document
import org.apache.lucene.document.Field
import org.apache.lucene.document.IntPoint
import org.apache.lucene.document.StoredField
import org.apache.lucene.document.StringField
import org.apache.lucene.index.Term
import org.apache.lucene.search.Query
import org.apache.lucene.search.TermQuery
import ru.yandex.travel.dicts.rasp.proto.TCountry
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField
import java.util.NoSuchElementException

class CountryService(
    params: CountryDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TCountry>
) : LuceneIndexService<TCountry>(params, luceneIndexBuilder, "DictCountry"), CountryDataProvider {
    companion object {
        private val FIELD_ID_INDEX = "ii"
        private val FIELD_GEO_ID_INDEX = "gii"
        private val FIELD_PROTO_STORE = "ps"
        private val FIELD_CODE_INDEX = "ci"
    }

    override fun getById(id: Int): TCountry {
        val query = IntPoint.newExactQuery(FIELD_ID_INDEX, id)

        return searchOne(query)?.toCountry()
            ?: throw NoSuchElementException("No such country for id: $id")
    }

    override fun getByGeoId(geoId: Int): TCountry {
        val query = IntPoint.newExactQuery(FIELD_GEO_ID_INDEX, geoId)

        return searchOne(query)?.toCountry()
            ?: throw NoSuchElementException("No such country for geoId: $geoId")
    }

    override fun getByCodeId(codeId: String): TCountry {
        val query: Query = TermQuery(Term(FIELD_CODE_INDEX, codeId))

        return searchOne(query)?.toCountry()
            ?: throw NoSuchElementException("No such country for geoId: $codeId")
    }

    private fun Document.toCountry(): TCountry {
        return getProtoField(TCountry.parser(), FIELD_PROTO_STORE)
    }

    override fun documentProducer(row: TCountry): Iterable<Document> {
        val document = Document()

        document.add(IntPoint(FIELD_ID_INDEX, row.id))
        document.add(IntPoint(FIELD_GEO_ID_INDEX, row.geoId))
        document.add(StringField(FIELD_CODE_INDEX, row.code, Field.Store.YES))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))

        return listOf(document)
    }
}
