package ru.yandex.travel.api.services.dictionaries.train.settlement

import org.apache.lucene.document.*
import ru.yandex.travel.dicts.rasp.proto.TSettlement
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField
import java.util.NoSuchElementException
import java.util.Optional

class TrainSettlementService(
    params: TrainSettlementDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TSettlement>
) : LuceneIndexService<TSettlement>(params, luceneIndexBuilder, "TrainSettlement"), TrainSettlementDataProvider {
    companion object {
        private val FIELD_ID_INDEX = "ii"
        private val FIELD_GEO_ID_INDEX = "gii"
        private val FIELD_PROTO_STORE = "ps"
    }

    override fun getById(id: Int): TSettlement {
        val query = IntPoint.newExactQuery(FIELD_ID_INDEX, id)

        return searchOne(query)?.toSettlement()
            ?: throw NoSuchElementException("No such settlement for id: $id")
    }

    override fun getOptionalSettlementByGeoId(geoId: Int): Optional<TSettlement> {
        val query = IntPoint.newExactQuery(FIELD_GEO_ID_INDEX, geoId)
        val result = searchOne(query) ?: return Optional.empty<TSettlement>()

        return Optional.of(result.toSettlement())
    }

    override fun getSettlementCode(settlementId: Int): String? {
        val settlement = getById(settlementId)

        if (!settlement.iata.equals("")) {
            return settlement.iata
        }

        if (!settlement.sirenaId.equals("")) {
            return settlement.sirenaId
        }

        return null
    }

    override fun documentProducer(row: TSettlement): Iterable<Document> {
        val document = Document()

        document.add(IntPoint(FIELD_ID_INDEX, row.id))
        document.add(IntPoint(FIELD_GEO_ID_INDEX, row.geoId))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))

        return listOf(document)
    }

    private fun Document.toSettlement(): TSettlement {
        return getProtoField(TSettlement.parser(), FIELD_PROTO_STORE)
    }
}
