package ru.yandex.travel.api.services.dictionaries.train.station

import org.apache.lucene.document.Document
import org.apache.lucene.document.IntPoint
import org.apache.lucene.document.StoredField
import ru.yandex.travel.dicts.rasp.proto.TStation
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField
import java.util.NoSuchElementException

class TrainStationService(
    params: TrainStationDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TStation>
) : LuceneIndexService<TStation>(params, luceneIndexBuilder, "TrainStation"), TrainStationDataProvider {
    companion object {
        private val FIELD_ID_INDEX = "ii"
        private val FIELD_SETTLEMENT_ID_INDEX = "settlementId"
        private val FIELD_PROTO_STORE = "ps"
    }

    override fun getById(id: Int): TStation {
        val query = IntPoint.newExactQuery(FIELD_ID_INDEX, id)

        return searchOne(query)?.toStation()
            ?: throw NoSuchElementException("No such station for id: $id")
    }

    override fun findBySettlementId(settlementId: Int): List<TStation> {
        return luceneIndex.search { searcher ->
            val query = IntPoint.newExactQuery(FIELD_SETTLEMENT_ID_INDEX, settlementId)
            val docs = searcher.search(query, 10000)

            if (docs.totalHits < 1) {
                return@search listOf()
            }

            return@search docs.scoreDocs.map { searcher.doc(it.doc).toStation() }
        }
    }

    override fun documentProducer(row: TStation): Iterable<Document> {
        val document = Document()

        document.add(IntPoint(FIELD_ID_INDEX, row.id))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))
        if (row.settlementId != 0 && !row.isHidden) {
            document.add(IntPoint(FIELD_SETTLEMENT_ID_INDEX, row.settlementId))
        }

        return listOf(document)
    }

    private fun Document.toStation(): TStation {
        return getProtoField(TStation.parser(), FIELD_PROTO_STORE)
    }
}
