package ru.yandex.travel.api.services.dictionaries.train.station_code

import org.apache.lucene.document.*
import org.apache.lucene.index.Term
import org.apache.lucene.search.TermQuery
import ru.yandex.travel.dicts.rasp.proto.ECodeSystem
import ru.yandex.travel.dicts.rasp.proto.TStationCode
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getIntField
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField

class TrainStationCodeService(
    params: TrainStationCodeDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TStationCode>
) : LuceneIndexService<TStationCode>(params, luceneIndexBuilder, "TrainStationCode"), TrainStationCodeDataProvider {
    companion object {
        private val FIELD_STATION_ID_INDEX = "sii"
        private val FIELD_STATION_ID_STORE = "sis"
        private val FIELD_CODE = "c"
        private val FIELD_PROTO_STORE = "ps"
    }

    override fun getStationExpressCode(stationId: Int): TStationCode {
        val query = IntPoint.newExactQuery(FIELD_STATION_ID_INDEX, stationId)

        return searchOne(query)?.toStationCode()
            ?: throw NoSuchElementException("No such stationCode for stationId: $stationId")
    }

    override fun getStationIdByCode(code: String): Int {
        val query = TermQuery(Term(FIELD_CODE, code))

        return searchOne(query)?.getIntField(FIELD_STATION_ID_STORE)
            ?: throw NoSuchElementException("No id found for code: $code")
    }

    override fun documentProducer(row: TStationCode): Iterable<Document> {
        if (row.systemId != ECodeSystem.CODE_SYSTEM_EXPRESS) {
            return emptyList()
        }

        val document = Document()

        document.add(IntPoint(FIELD_STATION_ID_INDEX, row.stationId))
        document.add(StoredField(FIELD_STATION_ID_STORE, row.stationId))
        document.add(StringField(FIELD_CODE, row.code, Field.Store.NO))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))

        return listOf(document)
    }

    private fun Document.toStationCode(): TStationCode {
        return getProtoField(TStationCode.parser(), FIELD_PROTO_STORE)
    }
}
