package ru.yandex.travel.api.services.dictionaries.train.station_to_settlement

import org.apache.lucene.document.Document
import org.apache.lucene.document.IntPoint
import org.apache.lucene.document.StoredField
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationService
import ru.yandex.travel.dicts.rasp.proto.TSettlement
import ru.yandex.travel.dicts.rasp.proto.TStation2Settlement
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField
import java.util.Optional

class TrainStationToSettlementService(
    params: TrainStationToSettlementDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TStation2Settlement>
) : LuceneIndexService<TStation2Settlement>(params, luceneIndexBuilder, "TrainStationToSettlement"),
    TrainStationToSettlementDataProvider {
    companion object {
        private const val FIELD_SETTLEMENT_ID_INDEX = "settlementId"
        private const val FIELD_PROTO_STORE = "ps"
    }

    override fun getStationIds(settlementId: Int): List<Int> {
        return luceneIndex.search { searcher ->
            val query = IntPoint.newExactQuery(FIELD_SETTLEMENT_ID_INDEX, settlementId)
            val docs = searcher.search(query, 10000)

            if (docs.totalHits < 1) {
                return@search listOf()
            }

            return@search docs.scoreDocs.map { searcher.doc(it.doc).toStationToSettlement().stationId }
        }
    }

    override fun documentProducer(row: TStation2Settlement): Iterable<Document> {
        val document = Document()

        document.add(IntPoint(FIELD_SETTLEMENT_ID_INDEX, row.settlementId))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))
        return listOf(document)
    }

    private fun Document.toStationToSettlement(): TStation2Settlement {
        return getProtoField(TStation2Settlement.parser(), FIELD_PROTO_STORE)
    }
}
