package ru.yandex.travel.api.services.geo;

import java.util.NoSuchElementException;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.api.services.geo.model.CrossSearchPointInfo;
import ru.yandex.travel.api.services.geo.model.PointId;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;
import ru.yandex.travel.dicts.rasp.proto.TStation;


@Component
@RequiredArgsConstructor
public class CrossSearchPointProvider {
    private final TrainStationDataProvider stationDataProvider;
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final GeoBase geoBase;

    public CrossSearchPointInfo getByPointKey(PointId pointId, String locale, String domain) {
        var pointType = PointProviderHelper.getPointType(pointId);
        var id = PointProviderHelper.parsePointId(pointId.getPointKey());
        switch (pointType) {
            case GEO:
                return getByGeoId(id, locale, domain);
            case STATION:
                return getByStation(pointId, locale, domain);
            case SETTLEMENT:
                return getBySettlementId(id, locale, domain);
            default:
                return null;
        }
    }

    private CrossSearchPointInfo getBySettlement(TSettlement settlement, String locale, String domain) {
        if (settlement == null) {
            return null;
        }
        var settlementGeoId = settlement.getGeoId();
        var builder = CrossSearchPointInfo
                .builder()
                .geoId(settlementGeoId)
                .pointKey(PointProviderHelper.buildSettlementPointKey(settlement.getId()));
        if (settlementGeoId != 0) {
            var settlementTitle =
                    GeoBaseHelpers.getRegionLinguistics(geoBase, settlement.getGeoId(), locale).getNominativeCase();
            builder = builder.title(settlementTitle);
        }
        var regionGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, settlementGeoId, GeoBaseHelpers.REGION_REGION_TYPE
                , domain);
        if (regionGeoId != null) {
            var regionTitle = GeoBaseHelpers.getRegionLinguistics(geoBase, regionGeoId, locale).getNominativeCase();
            builder = builder.regionTitle(regionTitle);
        }
        var countryGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, settlement.getGeoId(),
                GeoBaseHelpers.COUNTRY_REGION_TYPE, domain);
        if (countryGeoId != null) {
            var countryTitle = GeoBaseHelpers.getRegionLinguistics(geoBase, countryGeoId, locale).getNominativeCase();
            builder = builder.countryTitle(countryTitle);
        }
        builder.slug(settlement.getSlug());
        return builder.build();
    }

    private CrossSearchPointInfo getBySettlementId(int settlementId, String locale, String domain) {
        TSettlement settlement;
        try {
            settlement = trainSettlementDataProvider.getById(settlementId);
        } catch (NoSuchElementException e) {
            return null;
        }
        return getBySettlement(settlement, locale, domain);
    }

    private CrossSearchPointInfo getByStation(PointId pointId, String locale, String domain) {
        var stationId = PointProviderHelper.parsePointId(pointId.getPointKey());
        TStation station;
        try {
            station = stationDataProvider.getById(stationId);
        } catch (NoSuchElementException e) {
            return null;
        }
        var settlementId = station.getSettlementId();
        return getBySettlementId(settlementId, locale, domain);
    }

    public CrossSearchPointInfo getByGeoId(Integer geoId, String domain, String locale) {
        var settlement = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoId);
        var regionTypeId = GeoBaseHelpers.getRegionTypeId(geoBase, domain, geoId);
        if (settlement.isPresent() && regionTypeId == GeoBaseHelpers.CITY_REGION_TYPE) {
            return getBySettlement(settlement.get(), locale, domain);
        }
        if (regionTypeId < GeoBaseHelpers.CITY_REGION_TYPE) {
            return null;
        }
        var parentCityGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, geoId, GeoBaseHelpers.CITY_REGION_TYPE, domain);
        if (parentCityGeoId == null) {
            return null;
        }
        settlement = trainSettlementDataProvider.getOptionalSettlementByGeoId(parentCityGeoId);
        if (settlement.isEmpty()) {
            return null;
        }
        return getBySettlement(settlement.get(), locale, domain);
    }
}
