package ru.yandex.travel.api.services.geo;

import java.util.ArrayList;
import java.util.List;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.geo.model.PointId;
import ru.yandex.travel.api.services.geo.model.PointInfo;
import ru.yandex.travel.api.services.geo.typed_providers.CountryPointInfoProvider;
import ru.yandex.travel.api.services.geo.typed_providers.RegionPointInfoProvider;
import ru.yandex.travel.api.services.geo.typed_providers.SettlementPointInfoProvider;
import ru.yandex.travel.api.services.geo.typed_providers.StationPointInfoProvider;

import static ru.yandex.travel.api.services.geo.PointInfoHelper.getByIdErrorWrapper;


@Component
@RequiredArgsConstructor
class PointInfoProviderKernel {
    private final StationPointInfoProvider stationPointInfoProvider;
    private final SettlementPointInfoProvider settlementPointInfoProvider;
    private final RegionPointInfoProvider regionPointInfoProvider;
    private final CountryPointInfoProvider countryPointInfoProvider;

    public List<PointInfo> getByGeoId(int pointGeoId, String locale, String domain, boolean stopOnFirst) {
        Integer currentGeoId = pointGeoId;
        List<PointInfo> result = new ArrayList<>();

        List<PointInfoGetterFunction<Integer>> getters = List.of(
                settlementPointInfoProvider::getByGeoId,
                regionPointInfoProvider::getByGeoId,
                countryPointInfoProvider::getByGeoId
        );

        for (PointInfoGetterFunction<Integer> getter : getters) {
            PointInfo pointInfo = getByIdErrorWrapper(getter, currentGeoId, locale, domain);
            if (pointInfo != null) {
                currentGeoId = pointInfo.getParentGeoId();
                result.add(pointInfo);
                if (currentGeoId == null || stopOnFirst) {
                    return result;
                }
            }
        }

        return result;
    }

    public PointInfo getByPointKey(String pointKey, String locale, String domain) {
        PointId pointId = PointId.builder().pointKey(pointKey).build();
        var pointType = PointProviderHelper.getPointType(pointId);
        var intPointId = PointProviderHelper.parsePointId(pointId.getPointKey());
        switch (pointType) {
            case STATION:
                return stationPointInfoProvider.getById(intPointId, locale, domain);
            case SETTLEMENT:
                return settlementPointInfoProvider.getById(intPointId, locale, domain);
            case REGION:
                return regionPointInfoProvider.getById(intPointId, locale, domain);
            case COUNTRY:
                return countryPointInfoProvider.getById(intPointId, locale, domain);
            default:
                return null;
        }
    }
}
