package ru.yandex.travel.api.services.hotels.amenities;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;
import yandex.maps.proto.search.business.Business;

import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils;
import ru.yandex.travel.api.models.hotels.Amenity;
import ru.yandex.travel.api.models.hotels.AmenityGroup;
import ru.yandex.travel.api.models.hotels.Hotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelFeatureGroup;
import ru.yandex.travel.hotels.geosearch.model.GeoSimilarHotel;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsResponse;

@Component
@EnableConfigurationProperties(AmenityServiceProperties.class)
@Slf4j
public class AmenityService {
    private final Set<String> bannedFeatures;
    private final Map<String, Integer> mainFeatureOrder;
    private final Map<String, Integer> featureGroupOrder;
    private final Map<String, AmenityServiceProperties.GeoFeatureConfig> geoFeatures;

    public AmenityService(AmenityServiceProperties config) {
        bannedFeatures = ImmutableSet.copyOf(config.getBannedFeatures());

        var mainFeatureOrderBuilder = new ImmutableMap.Builder<String, Integer>();
        for (int i = 0; i < config.getMainFeatureOrder().size(); ++i) {
            mainFeatureOrderBuilder.put(config.getMainFeatureOrder().get(i), i);
        }
        mainFeatureOrder = mainFeatureOrderBuilder.build();

        var featureGroupOrderBuilder = new ImmutableMap.Builder<String, Integer>();
        for (int i = 0; i < config.getFeatureGroupOrder().size(); ++i) {
            featureGroupOrderBuilder.put(config.getFeatureGroupOrder().get(i), i);
        }
        featureGroupOrder = featureGroupOrderBuilder.build();

        geoFeatures = config.getGeoFeatures().stream()
                .collect(Collectors.toUnmodifiableMap(x -> x.getBusinessFeatureId(), x -> x));

        Preconditions.checkState(geoFeatures.values().stream().allMatch(x -> x.isSingleConfigDefined()),
                "All geoFeatures configs should have static or dynamic config");
    }

    private String transformName(String name) {
        if (name.isEmpty()) {
            return name;
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    private void sortAmenities(List<Amenity> amenities, Map<String, Integer> order) {
        amenities.sort((lhs, rhs) -> {
            Integer lhsOrder = order.getOrDefault(lhs.getId(), order.size());
            Integer rhsOrder = order.getOrDefault(rhs.getId(), order.size());
            return lhsOrder.compareTo(rhsOrder);
        });
    }

    public void fillForSimilarHotel(Hotel rspHotel, GeoSimilarHotel hotel) {
        List<Amenity> amenities = new ArrayList<>();
        if (hotel.getExtension() != null) {
            for (GeoSimilarHotel.Feature feature : hotel.getExtension().getFeatures()) {
                if (bannedFeatures.contains(feature.getId())) {
                    continue;
                }
                if (!feature.isMain()) {
                    // Для похожих отелей берем только главные фичи
                    continue;
                }
                String name = transformName(feature.getName());
                if (feature.getBoolValue() != null && feature.getBoolValue()) {
                    Amenity am = new Amenity();
                    am.setId(feature.getId());
                    am.setName(name);
                    amenities.add(am);
                }
                if (feature.getEnumValues() != null) {
                    for (String enumValue : feature.getEnumValues()) {
                        Amenity am = new Amenity();
                        am.setId(feature.getId());
                        am.setName(String.format("%s: %s", name, enumValue));
                        amenities.add(am);
                    }
                }
                if (feature.getStringValue() != null) {
                    Amenity am = new Amenity();
                    am.setId(feature.getId());
                    am.setName(String.format("%s: %s", name, feature.getStringValue()));
                    amenities.add(am);
                }
            }
        }
        sortAmenities(amenities, mainFeatureOrder);
        rspHotel.setMainAmenities(amenities);
    }

    public void fillForUsualHotel(Hotel rspHotel, GeoHotel geoHotel, boolean onlyMain) {
        // На входе имеем:
        // Список и порядок главных фичей (конфиг)
        // порядок групп (конфиг)
        // Список всех фичей (feature)
        // Опционально: Список групп фичей (featureGroups)
        Map<String/*FeatureId*/, AmenityGroup> featureId2group = new HashMap<>();
        List<AmenityGroup> groups = new ArrayList<>();
        if (!onlyMain && geoHotel.getFeatureGroups() != null) {
            for (GeoHotelFeatureGroup geoGroup : geoHotel.getFeatureGroups()) {
                AmenityGroup group = new AmenityGroup();
                group.setId(geoGroup.getId());
                group.setName(geoGroup.getName());
                group.setAmenities(new ArrayList<>());
                groups.add(group);
                for (String featureId : geoGroup.getFeatureIds()) {
                    featureId2group.put(featureId, group);
                }
            }
        }
        List<Amenity> mainAmenities = new ArrayList<>();
        for (Business.Feature geoFeature : geoHotel.getGeoObjectMetadata().getFeatureList()) {
            if (!geoFeature.hasValue() || !geoFeature.hasName() || !geoFeature.hasId()) {
                continue;
            }
            if (bannedFeatures.contains(geoFeature.getId())) {
                continue;
            }
            List<Amenity> amenities;
            if (mainFeatureOrder.containsKey(geoFeature.getId())) {
                amenities = mainAmenities;
            } else {
                AmenityGroup group = featureId2group.get(geoFeature.getId());
                if (group == null) {
                    continue;
                }
                amenities = group.getAmenities();
            }
            String name = transformName(geoFeature.getName());
            if (geoFeature.getValue().hasBooleanValue() && geoFeature.getValue().getBooleanValue()) {
                Amenity am = new Amenity();
                am.setId(geoFeature.getId());
                am.setName(name);
                amenities.add(am);
            }
            for (Business.EnumItem enumItem : geoFeature.getValue().getEnumValueList()) {
                Amenity am = new Amenity();
                am.setId(geoFeature.getId());
                am.setName(String.format("%s: %s", name, enumItem.getName()));
                amenities.add(am);
            }
            if (geoFeature.getValue().getTextValueCount() > 0) {
                // Take only first
                String textValue = geoFeature.getValue().getTextValue(0);
                Amenity am = new Amenity();
                am.setId(geoFeature.getId());
                am.setName(String.format("%s: %s", name, textValue));
                amenities.add(am);
            }
        }
        sortAmenities(mainAmenities, mainFeatureOrder);
        rspHotel.setMainAmenities(mainAmenities);
        groups = groups.stream().filter(g -> !g.getAmenities().isEmpty()).collect(Collectors.toList());
        groups.sort((lhs, rhs) -> {
            Integer lhsOrder = featureGroupOrder.getOrDefault(lhs.getId(), featureGroupOrder.size());
            Integer rhsOrder = featureGroupOrder.getOrDefault(rhs.getId(), featureGroupOrder.size());
            return lhsOrder.compareTo(rhsOrder);
        });
        rspHotel.setAmenityGroups(groups);
        rspHotel.setGeoFeature(buildGeoFeature(geoHotel.getGeoObjectMetadata().getFeatureList()));
    }

    public Hotel.HotelGeoFeature buildGeoFeature(List<Business.Feature> featureList) {
        @Data
        @AllArgsConstructor
        class GeoFeatureInfo {
            String id;
            String icon;
            String name;
            Integer importance;
        }

        var geoFeature = new Hotel.HotelGeoFeature();
        var firstGeoFeature = featureList.stream()
                .map(rspFeature -> {
                    if (!geoFeatures.containsKey(rspFeature.getId())) {
                        return null;
                    }
                    var geoFeatureConfig = geoFeatures.get(rspFeature.getId());
                    if (geoFeatureConfig.getStaticConfig() != null) {
                        var realValues = Sets.union(
                                rspFeature.getValue().getEnumValueList().stream().map(x -> x.getId())
                                        .collect(Collectors.toUnmodifiableSet()),
                                Set.of(rspFeature.getValue().getBooleanValue() ? "1" : "0")
                        );
                        var allowedValues = Set.copyOf(geoFeatureConfig.getStaticConfig().getBusinessFeatureValues());
                        if (!Sets.intersection(allowedValues, realValues).isEmpty()) {
                            return new GeoFeatureInfo(geoFeatureConfig.getId(), geoFeatureConfig.getIcon(),
                                    geoFeatureConfig.getStaticConfig().getName(), geoFeatureConfig.getImportance());
                        } else {
                            return null;
                        }
                    } else {
                        var dynamicConfig = geoFeatureConfig.getDynamicConfig();
                        if (rspFeature.getValue().getTextValueCount() != 1) {
                            return null;
                        }
                        int currentValue;
                        try {
                            currentValue = (int)Double.parseDouble(rspFeature.getValue().getTextValue(0));
                        } catch (NumberFormatException e) {
                            return null;
                        }
                        if (currentValue > dynamicConfig.getMaxValue()) {
                            return null;
                        }
                        var name = HotelsPortalUtils.formatDistanceMeters(currentValue) + " " + dynamicConfig.getNameSuffix();
                        return new GeoFeatureInfo(geoFeatureConfig.getId(), geoFeatureConfig.getIcon(),
                                name, geoFeatureConfig.getImportance());
                    }
                })
                .filter(Objects::nonNull)
                .max(Comparator.comparing(GeoFeatureInfo::getImportance));
        if (firstGeoFeature.isEmpty()) {
            return null;
        }
        geoFeature.setId(firstGeoFeature.get().getId());
        geoFeature.setIcon(firstGeoFeature.get().getIcon());
        geoFeature.setName(firstGeoFeature.get().getName());
        return geoFeature;
    }

    public List<Amenity> buildMainAmenities(List<TGetHotelsResponse.TFeature> featureList) {
        List<Amenity> mainAmenities = new ArrayList<>();
        for (TGetHotelsResponse.TFeature feature : featureList) {
            if (bannedFeatures.contains(feature.getId())) {
                continue;
            }
            if (mainFeatureOrder.containsKey(feature.getId())) {
                addGeoCounterFeatureToAmenities(feature, mainAmenities);
            }
        }
        sortAmenities(mainAmenities, mainFeatureOrder);
        return mainAmenities;
    }

    public void addGeoCounterFeatureToAmenities(TGetHotelsResponse.TFeature feature, List<Amenity> amenities) {
        String name = transformName(feature.getName());
        if (feature.getValueCase() == TGetHotelsResponse.TFeature.ValueCase.BOOLEANVALUE) {
            if (feature.getBooleanValue()) {
                Amenity am = new Amenity();
                am.setId(feature.getId());
                am.setName(name);
                amenities.add(am);
            }
        } else if (feature.getValueCase() == TGetHotelsResponse.TFeature.ValueCase.NUMERICVALUE) {
            Amenity am = new Amenity();
            am.setId(feature.getId());
            am.setName(String.format("%s: %s", name, feature.getNumericValue()));
            amenities.add(am);
        } else if (feature.getValueCase() == TGetHotelsResponse.TFeature.ValueCase.TEXTVALUES) {
            for (var value : feature.getTextValues().getValuesList()) {
                Amenity am = new Amenity();
                am.setId(feature.getId());
                am.setName(String.format("%s: %s", name, value.getName()));
                amenities.add(am);
            }
        }
    }
}
