package ru.yandex.travel.api.services.hotels.calendar_prices

import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import ru.yandex.travel.api.endpoints.hotels_portal.Experiments
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils
import ru.yandex.travel.api.infrastucture.TravelPreconditions
import ru.yandex.travel.api.models.hotels.Price
import ru.yandex.travel.api.models.hotels.interfaces.DebugOfferSearchParamsProvider
import ru.yandex.travel.api.models.hotels.interfaces.OfferSearchParamsProvider
import ru.yandex.travel.api.models.hotels.interfaces.RequestAttributionProvider
import ru.yandex.travel.api.models.hotels.interfaces.SearchContextWithAttribution
import ru.yandex.travel.api.services.hotels.geobase.GeoBase
import ru.yandex.travel.api.services.hotels.offercache.OffercacheService
import ru.yandex.travel.commons.experiments.UaasSearchExperiments
import ru.yandex.travel.commons.http.CommonHttpHeaders
import ru.yandex.travel.commons.rate.Throttler
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.common.Permalink
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum
import ru.yandex.travel.hotels.offercache.api.TComplexHotelId
import ru.yandex.travel.hotels.offercache.api.TOCCurrency
import ru.yandex.travel.hotels.offercache.api.TReadReq
import ru.yandex.travel.hotels.offercache.api.TReadResp
import ru.yandex.travel.hotels.proto.ERequestClass
import java.time.Duration
import java.time.LocalDate
import java.time.Period
import java.util.concurrent.CompletableFuture
import java.util.concurrent.TimeUnit
import kotlin.streams.toList

@Service
@EnableConfigurationProperties(CalendarPricesProperties::class)
open class CalendarPricesService(
    private val offercacheService: OffercacheService,
    private val geoBase: GeoBase,
    private val calendarPricesProperties: CalendarPricesProperties
) {
    data class BaseCalendarParams(
        val dateStart: LocalDate,
        val dateStop: LocalDate,
        val checkinDate: LocalDate?,
        val checkoutDate: LocalDate?,
        val permalink: Permalink,
    )

    data class OffercacheResponseForDate(
        val date: LocalDate,
        val readResponse: TReadResp?,
        val error: Throwable?,
    )

    private val currencyMap = mapOf(
        TOCCurrency.ECurrency.RUB to Price.Currency.RUB, TOCCurrency.ECurrency.USD to Price.Currency.USD
    )

    private val throttler = Throttler(
        calendarPricesProperties.throttlerProperties.rateLimit,
        calendarPricesProperties.throttlerProperties.semaphoreLimit,
        calendarPricesProperties.throttlerProperties.bucket,
        calendarPricesProperties.throttlerProperties.window,
    )

    fun getCalendarPrices(
        calendar: BaseCalendarParams,
        search: SearchContextWithAttribution,
        userCredentials: UserCredentials,
        headers: CommonHttpHeaders,
        experiments: Experiments,
        uaasSearchExperiments: UaasSearchExperiments,
        geoOrigin: GeoOriginEnum,
    ): CompletableFuture<CalendarPrices> = CompletableFuture.supplyAsync {
        getCalendarPricesNow(
            calendar,
            search,
            userCredentials,
            headers,
            experiments,
            uaasSearchExperiments,
            geoOrigin,
        )
    }

    fun getNextIterationPollingDelayMs(context: CalendarPricesSearchContext, experiments: Experiments): Int {
        return HotelsPortalUtils.getPollingIterationsDelayMs(
            calendarPricesProperties.pollingDelays,
            experiments,
            context.pollingIteration
        )
    }

    private fun getCalendarPricesNow(
        calendar: BaseCalendarParams,
        search: SearchContextWithAttribution,
        userCredentials: UserCredentials,
        headers: CommonHttpHeaders,
        experiments: Experiments,
        uaasSearchExperiments: UaasSearchExperiments,
        geoOrigin: GeoOriginEnum,
    ): CalendarPrices {
        TravelPreconditions.checkRequestArgument(
            calendar.dateStop.isAfter(calendar.dateStart),
            "dateStop ${calendar.dateStop} should be after dateStart  ${calendar.dateStart}"
        )
        val nightsToStay: Long = if (calendar.checkinDate == null && calendar.checkoutDate == null) 1L else
            Period.between(calendar.checkinDate, calendar.checkoutDate).days.toLong()
        TravelPreconditions.checkRequestArgument(
            nightsToStay > 0,
            "nightsToStay should be positive, check check-in check-out dates pair"
        )
        val daysInRequest = daysInRequest(calendar.dateStop, calendar.dateStart)
        TravelPreconditions.checkRequestArgument(
            daysInRequest <= calendarPricesProperties.daysPerSingleRequestLimit,
            "Too many days in single request $daysInRequest, " +
                "limit is ${calendarPricesProperties.daysPerSingleRequestLimit}"
        )

        val futures = calendar.dateStart.datesUntil(calendar.dateStop).map {
            val request = getSingleDayReadRequest(
                date = it,
                nightsToStay = nightsToStay,
                permalink = calendar.permalink,
                search = search,
                headers = headers,
                userCredentials = userCredentials,
                experiments = experiments,
                uaasSearchExperiments = uaasSearchExperiments,
                userRegion = HotelsPortalUtils.determineUserRegion(geoBase, headers, search),
                geoOrigin = geoOrigin,
            )
            offercacheService.read(request)
                .orTimeout(calendarPricesProperties.offercacheResponseTimeout.toMillis(), TimeUnit.MILLISECONDS)
                .handle { response, error -> OffercacheResponseForDate(it, response, error) }
        }.toList()
        val responses = futures.map { futureResponse -> futureResponse.get() }
        val prices = responses.asSequence().mapNotNull(fun(response: OffercacheResponseForDate): CalendarDayPrice? {
            if (response.error != null) {
                return CalendarDayPrice(date = response.date, error = response.error)
            }
            if (response.readResponse == null) {
                return null
            }
            val rsp: TReadResp = response.readResponse
            if (rsp.hotelsMap.isEmpty()) {
                return null
            }
            val minPrice = rsp.hotelsMap.values
                .flatMap { tHotel -> tHotel.pricesList.map { it.price } }
                .minOrNull() ?: return CalendarDayPrice(
                date = response.date,
                checkoutDate = response.date.plusDays(nightsToStay),
            )
            if (!currencyMap.contains(rsp.currency)) {
                return null
            }
            return CalendarDayPrice(
                date = response.date,
                checkoutDate = response.date.plusDays(nightsToStay),
                price = Price(
                    value = minPrice, currency = currencyMap[rsp.currency]
                )
            )
        }).toList()
        val allFinished =
            responses.mapNotNull { it.readResponse }.all { it.isFinished } && responses.mapNotNull { it.error }
                .isEmpty()
        return CalendarPrices(prices = prices, allFinished = allFinished)
    }

    private fun daysInRequest(
        dateStop: LocalDate, dateStart: LocalDate
    ): Int = Period.between(dateStart, dateStop).days

    private fun getSingleDayReadRequest(
        date: LocalDate,
        nightsToStay: Long,
        permalink: Permalink,
        search: SearchContextWithAttribution,
        headers: CommonHttpHeaders,
        userCredentials: UserCredentials,
        experiments: Experiments,
        uaasSearchExperiments: UaasSearchExperiments,
        userRegion: Int?,
        geoOrigin: GeoOriginEnum,
    ): TReadReq {

        val hasSearchParams = true
        val doNotSortUsingPlus = false
        val prepareWithoutDates = false

        val request = HotelsPortalUtils.prepareOfferCacheRequestParams(
            search as RequestAttributionProvider,
            headers,
            userCredentials,
            userRegion,
            search as OfferSearchParamsProvider,
            search as DebugOfferSearchParamsProvider,
            null,
            null,
            prepareWithoutDates,
            experiments,
            uaasSearchExperiments,
            "calendar_prices/",
            hasSearchParams,
            doNotSortUsingPlus,
        )

        val attributionBuilder = request.attribution.toBuilder()
        attributionBuilder.geoOrigin = geoOrigin.value
        attributionBuilder.geoClientId = "travel.portal"
        request.attribution = attributionBuilder.build()

        request.requestClass = ERequestClass.RC_CALENDAR
        request.checkInDate = date.toString()
        request.checkOutDate = date.plusDays(nightsToStay).toString()
        request.compactResponseForCalendar = true
        request.full = false
        request.allowOutdated = true
        request.addHotelId(TComplexHotelId.newBuilder().setPermalink(permalink.asLong()).build())
        return request.build()
    }

    fun acquireThrottler(): Boolean {
        return throttler.acquire(System.currentTimeMillis()) == Throttler.EDecision.PASS;
    }

    fun releaseThrottler() {
        return throttler.release()
    }
}
