package ru.yandex.travel.api.services.hotels.geobase;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import io.micrometer.core.lang.NonNull;
import lombok.Value;

import ru.yandex.geobase6.LinguisticsItem;
import ru.yandex.geobase6.LookupException;
import ru.yandex.geobase6.RegionHash;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SuggestRspV1;
import ru.yandex.travel.api.models.Linguistics;

public class GeoBaseHelpers {

    public static final int DELETED_REGION = -1;
    public static final int WORLD_REGION = 10000;
    public static final int MOSCOW_REGION = 213;
    public static final int LENINGRAD_REGION = 2;
    public static final int RUSSIA_REGION = 225;
    public static final int MOSCOW_DISTRICT = 1;
    public static final int LENINGRAD_DISTRICT = 10174;

    public static final int HIDDEN_REGION_TYPE = -1;
    public static final int OTHER_REGION_TYPE = 0;
    public static final int COUNTRY_REGION_TYPE = 3;
    public static final int FOREIGN_TERRITORY_REGION_TYPE = 12; // У некоторых регионах в родителях не страна, а это
    public static final int REGION_REGION_TYPE = 5;// Субъект федерации
    public static final int CITY_REGION_TYPE = 6;
    public static final int VILLAGE_REGION_TYPE = 7; // Село
    public static final int DISTRICT_REGION_TYPE = 10;

    private static final Set<Integer> CITY_REGION_COUNTRY_TYPES = ImmutableSet.of(CITY_REGION_TYPE, REGION_REGION_TYPE
            , COUNTRY_REGION_TYPE);

    public static final Set<Integer> DEFAULT_ALLOWED_REGION_TYPES = ImmutableSet.of(CITY_REGION_TYPE,
            REGION_REGION_TYPE, COUNTRY_REGION_TYPE);
    public static final Set<Integer> VILLAGE_ALLOWED_REGION_TYPES = ImmutableSet.of(DISTRICT_REGION_TYPE,
            CITY_REGION_TYPE, REGION_REGION_TYPE, COUNTRY_REGION_TYPE);

    public static boolean regionIsBig(int inputType) {
        return CITY_REGION_COUNTRY_TYPES.contains(inputType);
    }


    public static final Set<Integer> MOSCOW_BANNED_REGIONS = ImmutableSet.of(1);


    // https://doc.yandex-team.ru/lib/libgeobase5/concepts/region-types.html
    public static final HashMap<Integer, String> REGION_TYPES = new HashMap<>();

    static {
        REGION_TYPES.put(-1, "скрытый регион");
        REGION_TYPES.put(0, "прочие регионы");
        REGION_TYPES.put(1, "континент");
        REGION_TYPES.put(2, "регион");
        REGION_TYPES.put(3, "страна");
        REGION_TYPES.put(4, "федеральный округ");
        REGION_TYPES.put(5, "субъект федерации");
        REGION_TYPES.put(6, "город");
        REGION_TYPES.put(7, "село");
        REGION_TYPES.put(8, "район города");
        REGION_TYPES.put(9, "станция метро");
        REGION_TYPES.put(10, "район субъекта федерации");
        REGION_TYPES.put(11, "аэропорт");
        REGION_TYPES.put(12, "заморская территория");
        REGION_TYPES.put(13, "район города второго уровня");
        REGION_TYPES.put(14, "станция монорельса");
        REGION_TYPES.put(15, "сельское поселение");
    }

    @Value
    public static class RegionInfo {
        int geoId;
        int type;
        String name;
    }

    public static Integer getRegionTypeId(GeoBase geoBase, String domain, int geoId) {
        return geoBase.getRegionById(geoId, domain).getAttr("type").getInteger();
    }

    public static String getRegionName(GeoBase geoBase, String domain, int geoId) {
        return geoBase.getRegionById(geoId, domain).getAttr("name").getString();
    }

    public static String getRegionDescription(
            GeoBase geoBase,
            String domain,
            int geoId,
            boolean isReversePaths,
            boolean withName,
            Set<Integer> allowedTypes,
            Set<Integer> bannedRegionsId,
            boolean isRegionType) {

        if (geoId == MOSCOW_REGION) {
            bannedRegionsId.add(MOSCOW_DISTRICT);
        }

        if (geoId == LENINGRAD_REGION) {
            bannedRegionsId.add(LENINGRAD_DISTRICT);
        }

        var allowedTypesCopy = new HashSet<>(allowedTypes);
        List<String> parts = new ArrayList<>();
        var regionTypeId = getRegionTypeId(geoBase, domain, geoId);
        String regionType = REGION_TYPES.getOrDefault(regionTypeId, "объект поиска");

        if (withName) {
            parts.add(getRegionName(geoBase, domain, geoId));
        }

        geoId = geoBase.getParentId(geoId, domain);
        for (RegionInfo region : getRegionChain(geoBase, geoId, domain)) {
            if (!bannedRegionsId.contains(region.getGeoId())) {
                int type = region.getType();
                if (allowedTypesCopy.contains(type)) {
                    parts.add(region.getName());
                    allowedTypesCopy.remove(type);
                }
                if (allowedTypesCopy.isEmpty()) {
                    break;
                }
            }
        }
        parts = isReversePaths ? parts : Lists.reverse(parts);
        if (isRegionType) {
            parts.add(0, regionType);
        }

        return String.join(", ", parts);

    }


    public static List<SuggestRspV1.SuggestItem> findAndRenameDoubleSuggestDescription(
            List<SuggestRspV1.SuggestItem> suggestItems,
            GeoBase geoBase,
            String domain) {
        HashMap<String, List<String>> amountSuggestDescription = new HashMap<>();
        Set<String> itemsNeedAddedVillageDescription = new HashSet<>();
        Set<String> itemsNeedAddedRegionType = new HashSet<>();

        suggestItems.forEach(suggestItem -> {
            String key = suggestItem.getName() + suggestItem.getDescription();
            var suggestItemList = amountSuggestDescription.computeIfAbsent(key, k -> new ArrayList<>());
            suggestItemList.add(suggestItem.getId());
        });
        amountSuggestDescription.forEach((key, suggestItemList) -> {
            if (suggestItemList.size() > 1) {
                itemsNeedAddedVillageDescription.addAll(suggestItemList);
            }
        });

        amountSuggestDescription.clear();
        suggestItems.forEach(suggestItem -> {
            String key = suggestItem.getName();
            var suggestItemList = amountSuggestDescription.computeIfAbsent(key, k -> new ArrayList<>());
            suggestItemList.add(suggestItem.getId());
        });
        amountSuggestDescription.forEach((key, suggestItemList) -> {
            if (suggestItemList.size() > 1) {
                itemsNeedAddedRegionType.addAll(suggestItemList);
            }
        });
        suggestItems.forEach(suggestItem -> suggestItem.setDescription(getRegionDescription(
                geoBase,
                domain,
                suggestItem.getRedirectParams().getGeoId(),
                true,
                false,
                itemsNeedAddedVillageDescription.contains(suggestItem.getId())
                        ? VILLAGE_ALLOWED_REGION_TYPES
                        : DEFAULT_ALLOWED_REGION_TYPES,
                new HashSet<>(),
                itemsNeedAddedRegionType.contains(suggestItem.getId()))));
        return suggestItems;
    }


    public static Linguistics getRegionLinguistics(GeoBase geoBase, int geoId, String lang) {
        LinguisticsItem geoLing = geoBase.getLinguistics(geoId, lang);
        Linguistics ling = new Linguistics();
        ling.setNominativeCase(geoLing.getNominativeCase());
        ling.setGenitiveCase(geoLing.getGenitiveCase());
        ling.setDativeCase(geoLing.getDativeCase());
        ling.setPreposition(geoLing.getPreposition());
        ling.setPrepositionalCase(geoLing.getPrepositionalCase());
        ling.setLocativeCase(geoLing.getLocativeCase());
        ling.setDirectionalCase(geoLing.getDirectionalCase());
        ling.setAblativeCase(geoLing.getAblativeCase());
        ling.setAccusativeCase(geoLing.getAccusativeCase());
        ling.setInstrumentalCase(geoLing.getInstrumentalCase());
        return ling;
    }

    public static Integer getRegionRoundTo(GeoBase geoBase, int geoId, int roundToType, String domain) {
        for (RegionInfo region : getRegionChain(geoBase, geoId, domain)) {
            if (region.getType() == roundToType) {
                return region.getGeoId();
            }
        }
        return null;
    }

    public static int getPreferredGeoId(GeoBase geoBase, int geoId, Set<Integer> preferredRegionTypes, String domain) {
        for (RegionInfo region : getRegionChain(geoBase, geoId, domain)) {
            if (preferredRegionTypes.contains(region.getType())) {
                return region.getGeoId();
            }
        }
        return geoId;
    }

    public static Integer getRegionIdByLocationOrNull(GeoBase geoBase, double lat, double lon) {
        try {
            return geoBase.getRegionIdByLocation(lat, lon);
        } catch (Exception e) {// For example it may happen if geobase is disabled
            return null;
        }
    }


    public static Iterable<RegionInfo> getRegionChain(GeoBase geoBase, int geoId) {
        return getRegionChain(geoBase, geoId, "");
    }

    public static Iterable<RegionInfo> getRegionChain(GeoBase geoBase, int geoId, String domain) {
        class RegionIterator implements Iterator<RegionInfo> {

            private RegionInfo nextRegion = tryGetRegionInfo(geoId);
            private int seenRegionsCnt = 0;

            public boolean hasNext() {
                return nextRegion != null;
            }

            public RegionInfo next() {
                seenRegionsCnt++;
                Preconditions.checkState(seenRegionsCnt < 100, "Too long regions chain");
                RegionInfo currentRegion = nextRegion;
                if (nextRegion.geoId == WORLD_REGION || nextRegion.geoId == 0) {
                    // 0 is a strange geoId with parentGeoId=0, which causes endless loop
                    nextRegion = null;
                    return currentRegion;
                }
                int parentGeoId = geoBase.getParentId(currentRegion.getGeoId(), domain);
                nextRegion = tryGetRegionInfo(parentGeoId);
                return currentRegion;
            }

            private RegionInfo tryGetRegionInfo(int regionGeoId) {
                try {
                    return getRegionInfo(regionGeoId);
                } catch (LookupException e) {
                    return null;
                }
            }

            private RegionInfo getRegionInfo(int regionGeoId) {
                RegionHash hash = geoBase.getRegionById(regionGeoId, domain);
                Integer type = hash.getAttr("type").getInteger();
                String name = hash.getAttr("name").getString();
                if (type == FOREIGN_TERRITORY_REGION_TYPE) {
                    type = COUNTRY_REGION_TYPE;
                }
                return new RegionInfo(regionGeoId, type, name);
            }
        }

        class RegionIterable implements Iterable<RegionInfo> {
            @Override
            @NonNull
            public Iterator<RegionInfo> iterator() {
                return new RegionIterator();
            }
        }

        return new RegionIterable();
    }
}
