package ru.yandex.travel.api.services.hotels.geocounter;

import java.net.InetAddress;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;

import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.hotels.proto.TPingRpcReq;
import ru.yandex.travel.hotels.proto.geocounter_service.GeoCounterServiceV1Grpc;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(GeoCounterProperties.class)
public class GeoCounterConfiguration {
    private final GeoCounterProperties properties;

    @Bean("GeoCounterChannelSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties.getChannel());
        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean("GeoCounterProdChannelSupplier")
    public ChannelSupplier getProdChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(
                properties.getProdChannel() == null ? properties.getChannel() : properties.getProdChannel());
        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    private HAGrpcChannelFactory buildChannelFactory(ChannelSupplier channelSupplier) {
        return HAGrpcChannelFactory.Builder.newBuilder().withPingProducer(
                channel -> buildCompletableFuture(GeoCounterServiceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getIsReady() ? ChannelState.READY : ChannelState.NOT_READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(this::createChannel)
                .build();
    }

    @Bean("GeoCounterGrpcChannelFactory")
    public HAGrpcChannelFactory haGrpcChannelFactory(@Qualifier("GeoCounterChannelSupplier") ChannelSupplier channelSupplier) {
        return buildChannelFactory(channelSupplier);
    }

    @Bean("GeoCounterGrpcProdChannelFactory")
    public HAGrpcChannelFactory haGrpcProdChannelFactory(@Qualifier("GeoCounterProdChannelSupplier") ChannelSupplier channelSupplier) {
        return buildChannelFactory(channelSupplier);
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target) {
        var clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        var loggingClientInterceptor = new LoggingClientInterceptor(clientFqdn, target,
                Set.of(GeoCounterServiceV1Grpc.getPingMethod().getFullMethodName()));
        var defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(properties.getTimeout());
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor))
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }
}
