package ru.yandex.travel.api.services.hotels.geocounter;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.protobuf.UInt32Value;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.HotelAdditionalFilter;
import ru.yandex.travel.api.models.hotels.HotelFilter;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsReq;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsRsp;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterReq;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterRsp;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.SpeculativeRetryStrategy;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.proto.geocounter_service.GeoCounterServiceV1Grpc;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetCountsRequest;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetCountsResponse;
import ru.yandex.travel.hotels.proto.geocounter_service.TGetHotelsResponse;
import ru.yandex.travel.hotels.proto.hotel_filters.THotelFilter;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Component
public class GeoCounterService {
    public static class InvalidPollingTokenError extends RuntimeException {
        public InvalidPollingTokenError() {
            super();
        }
    }

    private final HAGrpcChannelFactory haGrpcChannelFactory;
    private final HAGrpcChannelFactory haGrpcProdChannelFactory;
    private final RetryRateLimiter rateLimiter;
    private final RetryRateLimiter getHotelsRateLimiter;
    private final Retry retryHelper;

    public GeoCounterService(@Qualifier("GeoCounterGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory,
                             @Qualifier("GeoCounterGrpcProdChannelFactory") HAGrpcChannelFactory haGrpcProdChannelFactory,
                             Retry retryHelper) {
        this.haGrpcChannelFactory = haGrpcChannelFactory;
        this.haGrpcProdChannelFactory = haGrpcProdChannelFactory;
        this.rateLimiter = new RetryRateLimiter(0.3);
        this.getHotelsRateLimiter = new RetryRateLimiter(0.3);
        this.retryHelper = retryHelper;
    }

    private GeoCounterServiceV1Grpc.GeoCounterServiceV1FutureStub getGeoCounterFutureStub(boolean useProdGeoCounter) {
        var factory = useProdGeoCounter ? haGrpcProdChannelFactory : haGrpcChannelFactory;
        return GeoCounterServiceV1Grpc.newFutureStub(factory.getRoundRobinChannel());
    }

    public CompletableFuture<GeoCounterGetHotelsRsp> getHotels(GeoCounterGetHotelsReq getHotelsReq) {
        Instant started = Instant.now();
        return retryHelper.withSpeculativeRetry("GeoCounterService::getHotels/" + getHotelsReq.getNameSuffix(),
                req -> buildCompletableFuture(getGeoCounterFutureStub(getHotelsReq.isUseProdGeoCounter()).getHotels(req)),
                getHotelsReq.getGetHotelsRequest(),
                SpeculativeRetryStrategy.<TGetHotelsResponse>builder()
                        .shouldRetryOnException(e -> {
                            var status = Status.fromThrowable(e);
                            var retryable = !List.of(Status.Code.INVALID_ARGUMENT, Status.Code.NOT_FOUND,
                                    Status.Code.ALREADY_EXISTS, Status.Code.PERMISSION_DENIED,
                                    Status.Code.FAILED_PRECONDITION, Status.Code.OUT_OF_RANGE,
                                    Status.Code.UNIMPLEMENTED, Status.Code.UNAUTHENTICATED).contains(status.getCode());
                            if (!retryable) {
                                log.error("Status {} is not retryable: {}", status.getCode(), status.toString());
                            }
                            return retryable;
                        })
                        .validateResult(resp -> {})
                        .timeout(Duration.ofSeconds(3))
                        .retryDelay(Duration.ofMillis(400))
                        .numRetries(5)
                        .build(),
                getHotelsRateLimiter)
                .handle((protoRsp, t) -> {
                    if (t != null) {
                        Throwable innerException = t.getCause();
                        if (innerException instanceof StatusRuntimeException && ((StatusRuntimeException) innerException).getTrailers() != null) {
                            TError grpcError = ((StatusRuntimeException) innerException).getTrailers().get(ServerUtils.METADATA_ERROR_KEY);
                            if (grpcError != null) {
                                if (grpcError.getCode() == EErrorCode.EC_INVALID_POLLING_TOKEN) {
                                    log.info(String.format("Invalid token error (%s): %s", getHotelsReq.getNameSuffix(), grpcError.getMessage()));
                                    return CompletableFuture.<TGetHotelsResponse>failedFuture(new InvalidPollingTokenError());
                                }
                                var error = String.format("Failed get hotels request (%s): %s", getHotelsReq.getNameSuffix(), grpcError.getMessage());
                                return CompletableFuture.<TGetHotelsResponse>failedFuture(new RuntimeException(error));
                            }
                        }
                        return CompletableFuture.<TGetHotelsResponse>failedFuture(t);
                    } else {
                        return CompletableFuture.completedFuture(protoRsp);
                    }
                })
                .thenCompose(x -> x)
                .thenApply(protoRsp -> {
                    if (protoRsp.hasError()) {
                        var error = String.format("Unexpected error field: it should be empty (%s): %s", getHotelsReq.getNameSuffix(), protoRsp.getError().getMessage());
                        throw new RuntimeException(error);
                    }
                    var protoHotels = protoRsp.getHotels();
                    GeoCounterGetHotelsRsp rsp = new GeoCounterGetHotelsRsp();
                    rsp.setHotels(protoHotels.getHotelsList());
                    rsp.setHotelResultCounts(new GeoCounterGetHotelsRsp.HotelResultCounts(
                            protoHotels.getHotelResultCounts().getHotelsOnCurrentPageCount(),
                            protoHotels.getHotelResultCounts().getHaveMoreHotels(),
                            protoHotels.getHotelResultCounts().getTotalHotelsCount()
                    ));
                    var searchParams = protoHotels.getSearchParams();
                    rsp.setGeoId(searchParams.getGeoId());
                    rsp.setBbox(BoundingBox.of(
                            Coordinates.of(searchParams.getBbox().getLowerLeft().getLat(),
                                    searchParams.getBbox().getLowerLeft().getLon()),
                            Coordinates.of(searchParams.getBbox().getUpperRight().getLat(),
                                    searchParams.getBbox().getUpperRight().getLon())
                    ));
                    rsp.setCheckInDate(LocalDate.parse(searchParams.getCheckInDate()));
                    rsp.setCheckOutDate(LocalDate.parse(searchParams.getCheckOutDate()));
                    rsp.setAges(Ages.fromString(searchParams.getAges()));
                    rsp.setSortType(searchParams.getSortType());
                    if (searchParams.hasSortOrigin()) {
                        rsp.setSortOrigin(Coordinates.of(searchParams.getSortOrigin().getLat(), searchParams.getSortOrigin().getLon()));
                    }
                    rsp.setHotelLocationUseful(protoHotels.getAggregatedHotelData().getHotelLocationUseful());
                    rsp.setHasBoyOffers(protoHotels.getAggregatedHotelData().getHasBoyOffers());


                    if (protoHotels.hasIncrementalPollingData()) {
                        rsp.setPollingContext(protoHotels.getIncrementalPollingData().getPollingIterationToken());
                        rsp.setPollingId(protoHotels.getIncrementalPollingData().getPollingId());
                    }
                    rsp.setPollingFinished(protoHotels.getPollingFinished());

                    rsp.setReqId(protoHotels.getDebugData().getRequestId());
                    rsp.setSessionId(protoHotels.getDebugData().getSessionId());

                    if (protoHotels.hasExperimentalData()) {
                        rsp.setCryptaDataAvailable(protoHotels.getExperimentalData().getCryptaDataAvailable());
                    }

                    if (protoHotels.hasLogData()) {
                        rsp.setLogData(protoHotels.getLogData());
                    }

                    rsp.setResponseTime(Duration.between(started, Instant.now()));

                    return rsp;
                });
    }

    public CompletableFuture<GeoCounterRsp> getCounts(GeoCounterReq geoCounterReq) {
        Instant started = Instant.now();
        return retryHelper.withSpeculativeRetry("GeoCounterService::getCounts/" + geoCounterReq.getNameSuffix(),
                req -> buildCompletableFuture(getGeoCounterFutureStub(geoCounterReq.isUseProdGeoCounter()).getCounts(req)),
                prepareProtoReq(geoCounterReq),
                SpeculativeRetryStrategy.<TGetCountsResponse>builder()
                        .shouldRetryOnException(e -> true)
                        .validateResult(resp -> {
                            if (!resp.hasCounts()) {
                                throw new RuntimeException("No counts in response");
                            }
                        })
                        .timeout(Duration.ofSeconds(3))
                        .retryDelay(Duration.ofMillis(400))
                        .numRetries(5)
                        .build(),
                rateLimiter)
                .thenApply(protoRsp -> {
                    GeoCounterRsp rsp = new GeoCounterRsp();
                    rsp.setCounts(protoRsp.getCounts());
                    rsp.setResponseTime(Duration.between(started, Instant.now()));
                    return rsp;
                });
    }

    private TGetCountsRequest prepareProtoReq(GeoCounterReq req) {
        TGetCountsRequest.Builder builder = TGetCountsRequest.newBuilder();

        builder.setLowerLeftLat(req.getBbox().getLeftDown().getLat());
        builder.setLowerLeftLon(req.getBbox().getLeftDown().getLon());
        builder.setUpperRightLat(req.getBbox().getUpRight().getLat());
        builder.setUpperRightLon(req.getBbox().getUpRight().getLon());
        builder.setCheckInDate(req.getCheckinDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        builder.setCheckOutDate(req.getCheckoutDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        builder.setAges(req.getAges());
        builder.setUseOfferBusDataFilters(req.isUseOfferBus());

        builder.addAllInitialFilterGroups(req.getInitialFilterGroups().stream()
                .map(filterGroup -> TGetCountsRequest.TInitialFilterGroup.newBuilder()
                        .setUniqueId(filterGroup.getUniqueId())
                        .addAllFilters(filterGroup.getFilters()
                                .stream()
                                .map(GeoCounterService::filterModelToPbOrNull)
                                .filter(Objects::nonNull)
                                .collect(Collectors.toUnmodifiableList()))
                        .build())
                .collect(Collectors.toUnmodifiableList()));

        var priceFilterBuilder = THotelFilter.TPriceValue.newBuilder();
        if (req.getFilterPriceFrom() != null) {
            priceFilterBuilder.setTotalPriceFrom(UInt32Value.newBuilder()
                    .setValue(req.getFilterPriceFrom())
                    .build());
        }
        if (req.getFilterPriceTo() != null) {
            priceFilterBuilder.setTotalPriceTo(UInt32Value.newBuilder()
                    .setValue(req.getFilterPriceTo())
                    .build());
        }
        builder.addInitialFilterGroups(TGetCountsRequest.TInitialFilterGroup.newBuilder()
                .setUniqueId("~price~")
                .addAllFilters(List.of(THotelFilter.newBuilder()
                                .setUniqueId("~price~")
                                .setFeatureId("~price~")
                                .setPriceValue(priceFilterBuilder.build())
                                .build()))
                .build());

        var additionalFilters = req.getAdditionalFilters()
                .stream()
                .map(filter -> {
                    var filterPb = filterModelToPbOrNull(filter.getFilter());
                    if (filterPb == null) {
                        return null;
                    }
                    return TGetCountsRequest.TAdditionalFilter.newBuilder()
                            .setGroupId(filter.getGroupId())
                            .setType(filterTypeToPb(filter.getType()))
                            .setFilter(filterPb)
                            .build();
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toUnmodifiableList());

        builder.addAllAdditionalFilters(additionalFilters);
        return builder.build();
    }

    public static THotelFilter filterModelToPbOrNull(HotelFilter filter) {
        var filterBuilder = THotelFilter.newBuilder();
        filterBuilder.setUniqueId(filter.getUniqueId());
        filterBuilder.setFeatureId(filter.getFeatureId());
        if (filter.getNumericValue() != null) {
            filterBuilder.setComparableValue(THotelFilter.TComparableValue.newBuilder()
                    .setMode(comparableModeToPb(filter.getNumericValue().getMode()))
                    .setValue(filter.getNumericValue().getValue())
                    .build());
        } else if (filter.getListValue() != null) {
            filterBuilder.setListValue(THotelFilter.TListValue.newBuilder()
                    .addAllValue(filter.getListValue().getValues())
                    .build());
        } else if (filter.getIgnoredValue() != null) {
            filterBuilder.setIgnoredValue(THotelFilter.TIgnoredValue.newBuilder().build());
        } else {
            log.error("Filter has no numeric value nor list one. FeatureId: {}", filter.getFeatureId());
            return null;
        }
        return filterBuilder.build();
    }

    private static TGetCountsRequest.TAdditionalFilter.EType filterTypeToPb(HotelAdditionalFilter.Type type) {
        switch (type) {
            case or:
                return TGetCountsRequest.TAdditionalFilter.EType.Or;
            case and:
                return TGetCountsRequest.TAdditionalFilter.EType.And;
            case single:
                return TGetCountsRequest.TAdditionalFilter.EType.Single;
            default:
                throw new IllegalStateException(String.format("Unknown filter type: %s", type));
        }
    }

    private static THotelFilter.TComparableValue.EMode comparableModeToPb(HotelFilter.NumericValue.Mode mode) {
        switch (mode) {
            case less:
                return THotelFilter.TComparableValue.EMode.Less;
            case greater:
                return THotelFilter.TComparableValue.EMode.Greater;
            case lessOrEqual:
                return THotelFilter.TComparableValue.EMode.LessOrEqual;
            case greaterOrEqual:
                return THotelFilter.TComparableValue.EMode.GreaterOrEqual;
            default:
                throw new IllegalStateException(String.format("Unknown numeric value mode: %s", mode));
        }
    }
}
