package ru.yandex.travel.api.services.hotels.hotel_images;

import java.util.Collection;
import java.util.function.Function;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.proto.THotelImage;
import ru.yandex.travel.yt_lucene_index.MultiMapPersistentConfig;

@Component
@EnableConfigurationProperties(HotelImagesServiceProperties.class)
@Slf4j
public class HotelImagesService {

    private MultiMapPersistentConfig<Permalink, THotelImage, THotelImage> persistentConfig;

    public HotelImagesService(HotelImagesServiceProperties params) {
        if (params.isEnabled()) {
            persistentConfig = new MultiMapPersistentConfig<>(params, "HotelImages",
                    THotelImage::newBuilder, (x) -> Permalink.of(x.getPermalink()),
                Function.identity(), Function.identity());
        } else {
            persistentConfig = null;
        }
    }

    private void ensureEnabled() {
        if (persistentConfig == null) {
            throw new IllegalStateException("HotelImages service disabled");
        }
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }

    public Collection<THotelImage> getImagesByPermalink(Permalink permalink) {
        if (persistentConfig == null) {
            return null;
        }
        return persistentConfig.getByKey(permalink);
    }
}
