package ru.yandex.travel.api.services.hotels.legacy;

import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;


@Component
@ConditionalOnProperty(value = "hotels.legacy.index.type", havingValue = "lucene")
@EnableConfigurationProperties(LegacyHotelInfoLuceneProviderProperties.class)
@Slf4j
public class LegacyHotelInfoLuceneProvider implements LegacyHotelInfoProvider {
    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_TRAVEL_ID = "t";
    private static final String FIELD_RATING = "r";
    private static final String FIELD_PERMALINK = "p";
    private static final String FIELD_NAME = "n";
    private static final String FIELD_COUNTRY = "c";
    private static final String FIELD_IMAGE = "i";
    private static final String FIELD_STARS = "s";
    private static final String FIELD_ADDRESS = "a";
    private static final String FIELD_MEDIAN_PRICE = "m";

    private final YtLuceneIndex index;

    public LegacyHotelInfoLuceneProvider(LegacyHotelInfoLuceneProviderProperties params) {
        index = new YsonYtLuceneIndex(params, "LegacyHotels", INDEX_STRUCTURE_VERSION, (row) -> {
            Document document = new Document();
            for (YTreeNode listNode : row.getList("travel_ids")) {
                document.add(new LongPoint(FIELD_TRAVEL_ID, listNode.longValue()));
            }
            var rating = row.getDoubleO("rating");
            if (rating.isPresent()) {
                document.add(new StoredField(FIELD_RATING, rating.get()));
            }
            var permalink = row.getLongO("permalink");
            if (permalink.isPresent()) {
                document.add(new StoredField(FIELD_PERMALINK, permalink.get()));
            }
            document.add(new StoredField(FIELD_NAME, row.getString("name")));
            document.add(new StoredField(FIELD_COUNTRY, row.getString("country")));
            document.add(new StoredField(FIELD_IMAGE, row.getString("image")));
            var stars = row.getLongO("stars");
            if (stars.isPresent()) {
                document.add(new StoredField(FIELD_STARS, stars.get()));
            }
            var address = row.getStringO("address");
            if (address.isPresent()) {
                document.add(new StoredField(FIELD_ADDRESS, address.get()));
            }
            var medianPrice = row.getDoubleO("median_price");
            if (medianPrice.isPresent()) {
                document.add(new StoredField(FIELD_MEDIAN_PRICE, medianPrice.get()));
            }
            return Collections.singletonList(document);
        });
    }

    @PostConstruct
    public void init() {
        index.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        index.stop();
    }

    @Override
    public boolean isReady() {
        return index.isReady();
    }

    @Override
    public LegacyHotelInfo getHotelInfo(long travelId) {
        return index.search(searcher -> {
            Query query = LongPoint.newExactQuery(FIELD_TRAVEL_ID, travelId);
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits == 0) {
                return null;
            }
            Document doc = searcher.doc(topDocs.scoreDocs[0].doc);
            LegacyHotelInfo hi = new LegacyHotelInfo();
            hi.setName(doc.getField(FIELD_NAME).stringValue());
            var address = doc.getField(FIELD_ADDRESS);
            if (address != null) {
                hi.setAddress(address.stringValue());
            }
            hi.setCountry(doc.getField(FIELD_COUNTRY).stringValue());
            hi.setImage(doc.getField(FIELD_IMAGE).stringValue());
            var permalink = doc.getField(FIELD_PERMALINK);
            if (permalink != null) {
                hi.setPermalink(Permalink.of(permalink.stringValue()));
            }
            var medianPrice = doc.getField(FIELD_MEDIAN_PRICE);
            if (medianPrice != null) {
                hi.setMedianPrice(medianPrice.numericValue().doubleValue());
            }
            var stars = doc.getField(FIELD_STARS);
            if (stars != null) {
                hi.setStars(stars.numericValue().intValue());
            }
            var rating = doc.getField(FIELD_RATING);
            if (rating != null) {
                hi.setRating(rating.numericValue().doubleValue());
            }
            return hi;
        });
    }
}
