package ru.yandex.travel.api.services.hotels.legal_info;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.administrator.export.proto.HotelLegalInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.yt_lucene_index.NativeProtobufPlainYtLuceneIndex;


@Component
@EnableConfigurationProperties(LegalInfoDictionaryProperties.class)
public class HotelLegalInfoDictionary implements HealthIndicator {

    private static final String DICTIONARY_NAME = "HotelsLegalInfo";

    private final AtomicReference<Map<EPartnerId, Map<String, HotelLegalInfo>>> hotelsLegalInfoRef =
            new AtomicReference<>(null);
    private final AtomicLong tableCacheUpdateTimestamp = new AtomicLong(0);

    private final NativeProtobufPlainYtLuceneIndex<HotelLegalInfo> luceneIndex;

    public HotelLegalInfoDictionary(LegalInfoDictionaryProperties params) {
        luceneIndex = new NativeProtobufPlainYtLuceneIndex<>(params, DICTIONARY_NAME, HotelLegalInfo.class);
        luceneIndex.setIndexUpdateHandler(() -> {
            Map<EPartnerId, Map<String, HotelLegalInfo>> hotelLegalInfos = new HashMap<>();
            luceneIndex.forEachProtoRecord(HotelLegalInfo.parser(), proto -> {
                Map<String, HotelLegalInfo> hotelIdToLegalInfoMap = hotelLegalInfos
                        .computeIfAbsent(proto.getPartnerId(), partner -> new HashMap<>());
                if (hotelIdToLegalInfoMap.containsKey(proto.getHotelId())) {
                    throw new RuntimeException("It is expected to have only 1 info per hotel, but found 2. Partner: " + proto.getPartnerId() + "; HotelId: " + proto.getHotelId());
                }
                hotelIdToLegalInfoMap.put(proto.getHotelId(), proto);
            });
            hotelsLegalInfoRef.set(hotelLegalInfos);
            tableCacheUpdateTimestamp.set(Instant.now().toEpochMilli());
        });
    }

    @PostConstruct
    public void init() {
        luceneIndex.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        luceneIndex.stop();
    }

    public HotelLegalInfo getHotelLegalInfo(EPartnerId partnerId, String hotelId) {
        if (!hotelsLegalInfoRef.get().containsKey(partnerId)) {
            return null;
        }
        return hotelsLegalInfoRef.get().get(partnerId).get(hotelId);
    }

    public boolean isReady() {
        return hotelsLegalInfoRef.get() != null;
    }

    @Override
    public Health health() {
        if (isReady()) {
            return Health.up().build();
        } else {
            return Health.down().build();
        }
    }
}
