package ru.yandex.travel.api.services.hotels.min_prices;

import java.util.Collections;
import java.util.Optional;
import java.util.OptionalInt;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(MinMaxPricesServiceProperties.class)
@Slf4j
public class MinMaxPricesService {

    private static final int INDEX_STRUCTURE_VERSION = 4;

    private static final String FIELD_PERMALINK = "p";
    private static final String FIELD_MIN_PRICE = "m";
    private static final String FIELD_MAX_PRICE = "n";

    private final YtLuceneIndex minMaxPricesLuceneIndex;

    private final Counter minPricesByPermalinkFoundCounter;
    private final Counter maxPricesByPermalinkFoundCounter;
    private final Counter minPricesByPermalinkNotFoundCounter;
    private final Counter maxPricesByPermalinkNotFoundCounter;

    public MinMaxPricesService(MinMaxPricesServiceProperties params) {
        if (params.isEnabled()) {
            minMaxPricesLuceneIndex = new YsonYtLuceneIndex(params, "MinMaxPrices", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final long permalink = row.getLong("Permalink");
                document.add(new LongPoint(FIELD_PERMALINK, permalink));


                final Optional<Integer> minPrice = row.getIntO("SingleNightMedianMinPrice");
                final Optional<Integer> maxPrice = row.getIntO("SingleNightMedianMaxPrice");
                if (minPrice.isEmpty() && maxPrice.isEmpty()) {
                    return Collections.emptyList();
                }

                final int minPriceInt = minPrice.orElse(-1);
                final int maxPriceInt = maxPrice.orElse(-1);
                document.add(new StoredField(FIELD_MIN_PRICE, minPriceInt));
                document.add(new StoredField(FIELD_MAX_PRICE, maxPriceInt));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Min/max prices index disabled");
            minMaxPricesLuceneIndex = null;
        }
        minPricesByPermalinkFoundCounter =
                Counter.builder("min_prices.by_permalink.found").register(Metrics.globalRegistry);
        minPricesByPermalinkNotFoundCounter =
                Counter.builder("min_prices.by_permalink.notFound").register(Metrics.globalRegistry);
        maxPricesByPermalinkFoundCounter =
                Counter.builder("max_prices.by_permalink.found").register(Metrics.globalRegistry);
        maxPricesByPermalinkNotFoundCounter =
                Counter.builder("max_prices.by_permalink.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (minMaxPricesLuceneIndex != null) {
            minMaxPricesLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (minMaxPricesLuceneIndex != null) {
            minMaxPricesLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return minMaxPricesLuceneIndex == null || minMaxPricesLuceneIndex.isReady();
    }

    public OptionalInt getMinPriceByPermalink(long permalink) {
        OptionalInt result;
        if (minMaxPricesLuceneIndex != null) {
            result = getValue(permalink, FIELD_MIN_PRICE);
        } else {
            result = OptionalInt.empty();
        }
        if (result.isPresent()) {
            minPricesByPermalinkFoundCounter.increment();
        } else {
            minPricesByPermalinkNotFoundCounter.increment();
        }
        return result;
    }

    public OptionalInt getMaxPriceByPermalink(long permalink) {
        OptionalInt result;
        if (minMaxPricesLuceneIndex != null) {
            result = getValue(permalink, FIELD_MAX_PRICE);
        } else {
            result = OptionalInt.empty();
        }
        if (result.isPresent()) {
            minPricesByPermalinkFoundCounter.increment();
        } else {
            minPricesByPermalinkNotFoundCounter.increment();
        }
        return result;
    }

    private OptionalInt getValue(long permalink, String field) {
        return minMaxPricesLuceneIndex.search(searcher -> {
            Query query = LongPoint.newExactQuery(FIELD_PERMALINK, permalink);
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                var value = OptionalInt.of(document.getField(field).numericValue().intValue());
                if (value.getAsInt() != -1) {
                    return value;
                } else {
                    return OptionalInt.empty();
                }
            } else {
                return OptionalInt.empty();
            }
        });
    }

}


