package ru.yandex.travel.api.services.hotels.min_prices;

import java.util.Collections;
import java.util.Optional;
import java.util.OptionalInt;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.IntPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(RegionMinPricesServiceProperties.class)
@Slf4j
public class RegionMinPricesService {

    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_GEO_ID = "g";
    private static final String FIELD_MIN_PRICE = "m";

    private final YtLuceneIndex minPricesLuceneIndex;

    private final Counter regionMinPricesByGeoIdFoundCounter;
    private final Counter regionMinPricesByGeoIdNotFoundCounter;

    public RegionMinPricesService(RegionMinPricesServiceProperties params) {
        if (params.isEnabled()) {
            minPricesLuceneIndex = new YsonYtLuceneIndex(params, "RegionMinPrices", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final int permalink = row.getInt("GeoId");
                document.add(new IntPoint(FIELD_GEO_ID, permalink));

                final Optional<Integer> minPrice = row.getIntO("MinMedianMinPrice");
                if (!minPrice.isPresent()) {
                    return Collections.emptyList();
                }

                final int minPriceInt = minPrice.get();
                document.add(new StoredField(FIELD_MIN_PRICE, minPriceInt));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Region min prices index disabled");
            minPricesLuceneIndex = null;
        }
        regionMinPricesByGeoIdFoundCounter =
                Counter.builder("region_min_prices.by_geo_id.found").register(Metrics.globalRegistry);
        regionMinPricesByGeoIdNotFoundCounter =
                Counter.builder("region_min_prices.by_geo_id.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (minPricesLuceneIndex != null) {
            minPricesLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (minPricesLuceneIndex != null) {
            minPricesLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return minPricesLuceneIndex == null || minPricesLuceneIndex.isReady();
    }

    public OptionalInt getPriceByGeoId(int geoId) {
        OptionalInt result;
        if (minPricesLuceneIndex != null) {
            result = minPricesLuceneIndex.search(searcher -> {
                Query query = IntPoint.newExactQuery(FIELD_GEO_ID, geoId);
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return OptionalInt.of(document.getField(FIELD_MIN_PRICE).numericValue().intValue());
                } else {
                    return OptionalInt.empty();
                }
            });
        } else {
            result = OptionalInt.empty();
        }
        if (result.isPresent()) {
            regionMinPricesByGeoIdFoundCounter.increment();
        } else {
            regionMinPricesByGeoIdNotFoundCounter.increment();
        }
        return result;
    }
}


